# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import hashlib
import json
from collections import OrderedDict
from datetime import datetime
from enum import Enum

from six import text_type

from travel.cpa.lib.lib_datetime import timestamp
from travel.library.python.schematized import Schematized
from travel.library.python.schematized.fields import Any, Boolean, Enumeration, Float, String, UInt64, Int64


class OrderCurrencyCode(Enum):
    AZN = 'AZN'
    EUR = 'EUR'
    RUB = 'RUB'
    USD = 'USD'
    CNY = 'CNY'
    TRY = 'TRY'
    KZT = 'KZT'
    UAH = 'UAH'
    CHF = 'CHF'
    BYN = 'BYN'
    KRW = 'KRW'
    PLN = 'PLN'
    GBP = 'GBP'
    AMD = 'AMD'
    ILS = 'ILS'
    UZS = 'UZS'
    KGS = 'KGS'


class OrderStatus(Enum):
    UNPAID = 'unpaid'
    REFUNDED = 'refunded'
    PENDING = 'pending'
    PAID = 'paid'
    CONFIRMED = 'confirmed'
    CANCELLED = 'cancelled'
    DELETED = 'deleted'


class ItemState(Enum):
    NEW = 'new'
    RESERVED = 'reserved'
    CONFIRMED = 'confirmed'
    CANCELLED = 'cancelled'
    REFUNDED = 'refunded'
    PENDING = 'pending'


class DeferredPaymentEligibility(Enum):
    UNKNOWN = 'unknown'
    ELIGIBLE = 'eligible'
    NON_ELIGIBLE = 'non-eligible'


class OrderSnapshot(Schematized):
    __fields__ = OrderedDict([
        ('_timestamp', UInt64()),
        ('hash', String(optional=True)),
        ('travel_order_id', String()),
        ('partner_order_id', String()),
        ('status', Enumeration(OrderStatus, String)),
        ('label', String(optional=True)),
        ('currency_code', Enumeration(OrderCurrencyCode, String)),
        ('order_amount', Float()),
        ('profit_amount', Float()),
        ('created_at', UInt64()),
        ('updated_at', UInt64()),
        ('last_seen', UInt64(default=0)),
    ])

    def __init__(self):
        super(OrderSnapshot, self).__init__()
        self._timestamp = timestamp(datetime.utcnow())

    def __setattr__(self, key, value):
        if key == 'partner_name':
            raise AttributeError('This field is readonly')
        if key == 'partner_order_id':
            raise AttributeError('Do not set this field directly. Use "update_partner_order_id" instead')
        super(OrderSnapshot, self).__setattr__(key, value)

    def as_dict(self, validate=True):
        self.update_hash()
        return super(OrderSnapshot, self).as_dict(validate)

    def update_hash(self):
        m = hashlib.md5()
        for key, value in self.__values__.items():
            if key in ('hash', '_timestamp', 'updated_at', 'last_seen'):
                continue
            if value is None:
                continue
            if isinstance(self.__fields__[key], Any):
                text = json.dumps(value, separators=(',', ':'), sort_keys=True)
            else:
                text = repr(value)
            m.update(text.encode())
        self.hash = text_type(m.hexdigest())

    def update_partner_order_id(self, value):
        super(OrderSnapshot, self).__setattr__('travel_order_id', '{}:{}'.format(self.partner_name, value))
        super(OrderSnapshot, self).__setattr__('partner_order_id', value)


class GenericOrderSnapshot(OrderSnapshot):
    __fields__ = OrderedDict([
        ('category', String('generic')),
        ('partner_name', String('boy')),
        ('display_type', String()),
    ])


class AviaOrderSnapshot(OrderSnapshot):
    __fields__ = OrderedDict([
        ('category', String('avia')),
        ('source', String()),
        ('partner_id', UInt64()),
        ('billing_order_id', UInt64()),
        ('profit_amount', Float(optional=True)),
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('trip_type', String(optional=True)),
        ('date_forward', String(optional=True)),
        ('date_backward', String(optional=True)),
        ('pnr', String(optional=True)),
    ])


class NemoAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('source', String('nemo')),
        ('partner_profit', Float(optional=True))
    ])


class HotelsOrderSnapshot(OrderSnapshot):
    __fields__ = OrderedDict([
        ('category', String('hotels')),
        ('snapshot_source', String('collectors', optional=True)),
        ('partner_status', String(optional=True)),
        ('check_in', String()),
        ('check_out', String()),
        ('hotel_name', String()),
        ('hotel_country', String(optional=True, empty_is_none=True)),
        ('hotel_city', String(optional=True, empty_is_none=True)),
        ('promo_codes', String(optional=True)),
        ('promo_actions', String(optional=True)),
        ('discount_amount', Float(optional=True)),
        ('promocode_discount_amount', Float(optional=True)),
        ('strike_through_discount_amount', Float(optional=True)),
        ('privetmir_oct2020_eligible', Boolean(optional=True)),
        ('order_amount_payable', Float(optional=True)),
        ('use_deferred_payment', Boolean(optional=True)),
        ('post_pay_eligible', Boolean(default=False)),
        ('post_pay_used', Boolean(default=False)),
        ('amount_received_from_user', Float(optional=True)),
        ('refund_reason', String(optional=True)),
        ('deferred_payment_eligibility', Enumeration(DeferredPaymentEligibility, String,
                                                     default=DeferredPaymentEligibility.UNKNOWN)),
        ('initial_payment_amount', Float(optional=True)),
        ('penalty_amount_if_unpaid', Float(optional=True)),
        ('last_payment_scheduled_at', UInt64(optional=True)),
        ('fully_paid_at', UInt64(optional=True)),
        ('partner_hotel_id', String(optional=True)),
        ('yandex_plus_mode', String(optional=True)),
        ('yandex_plus_withdraw_points', Int64(optional=True)),
        ('yandex_plus_topup_points', Int64(optional=True)),
        ('yandex_plus_topup_date', UInt64(optional=True)),
        ('yandex_plus_user_balance', Int64(optional=True)),
        ('white_label_points_type', String(optional=True)),
        ('white_label_points_amount', Int64(optional=True)),
        ('white_label_customer_number', String(optional=True)),
    ])


class TrainOrder(Schematized):
    __fields__ = OrderedDict([
        ('adult_passengers_count', UInt64(description='Количество взрослых пассажиров')),
        ('active_ticket_count', UInt64(optional=True, description='Количество не возвращенных билетов')),
        ('arrival', UInt64(optional=True, description='Время прибытия, Timestamp UTC')),
        ('bandit_fee_applied_count', UInt64(optional=True, description='Количество комиссий бандита')),
        ('bandit_type', String(optional=True, description='Тип бандита вернувший наценку')),
        ('requested_bandit_type', String(optional=True, description='Тип бандита запрошенный с фронта')),
        ('bandit_version', UInt64(optional=True, description='Версия бандита')),
        ('bought_insurance_count', UInt64(description='Количество купленных страховок')),
        ('children_with_seats_count', UInt64(description='Количество детей с местами')),
        ('children_without_seats_count', UInt64(description='Количество детей без места')),
        ('coach_number', String(optional=True, description='Номер вагона')),
        ('coach_owner', String(optional=True, description='Перевозчик (carrier) от ИМ')),
        ('coach_type', String(optional=True, description='Тип вагона '
                                                         'platzkarte/compartment/suite/common/sitting/soft')),
        ('departure', UInt64(description='Время отправления, Timestamp UTC')),
        ('finished_at', UInt64(optional=True, description='Время завершения покупки, Timestamp UTC')),
        ('gender', String(optional=True, description='Гендерный признак купе')),
        ('insurance_auto_return', Boolean(description='Признак автовозврата страховок')),
        ('non_refundable_tickets_count', UInt64(description='Количество невозвратных билетов', default=0)),
        ('order_number', String(optional=True, description='Номер заказа РЖД (ReservationNumber)')),
        ('orders_created', String(optional=True, description='Идентификаторы ранее созданных заказов из order_history. '
                                                             'Для im_boy - пусто.')),
        ('offer_id', String(optional=True, description='yandex_uid')),
        ('partner_data_is_suburban', Boolean(optional=True, description='Признак электрички')),
        ('partner_data_operation_id', String(optional=True, description='Идентификатор покупки в ИМ operation_id')),
        ('partner_data_im_order_id', UInt64(optional=True, description='Идентификатор заказа в ИМ')),
        ('partner_data_provider', String(optional=True, description='Provider из ИМ (P1/P2)')),
        ('passengers_count', UInt64(optional=True, description='Количество пассажиров')),
        ('payment_attempts', UInt64(description='Было попыток оплат')),
        ('payment_purchase_token', String(optional=True, description='')),
        ('payment_status', String(optional=True, description='Статус платежа, для im и im_boy различаются.')),
        ('payment_ts', UInt64(optional=True, description='Время оплаты, Timestamp UTC')),
        ('payment_uid', String(optional=True, description='Внутренний UID платежа.')),
        ('payment_use_deferred_clearing', Boolean(description='Флаг включенного механизма отложенного клиринга')),
        ('permille', UInt64(optional=True, description='Наценка в 1/1000')),
        ('rebooked', Boolean(description='Признак факта перебронирования', default=False)),
        ('rebooking_available', Boolean(description='Флаг включенного механизма перебронирования')),
        ('refunded_ticket_count', UInt64(description='Количество возвращенных билетов.')),
        ('refunds_count', UInt64(description='Количество возвратов (без автовозврата страховки).')),
        ('requested_ticket_count', UInt64(description='Количество запрошенных для бронирования билетов')),
        ('service_class', String(optional=True, description='Код класса обслуживания')),
        ('scheme_id', String(optional=True, description='Идентификатор схемы вагона')),
        ('source_device', String(optional=True)),
        ('source_from', String(optional=True)),
        ('source_gclid', String(optional=True)),
        ('source_req_id', String(optional=True)),
        ('source_terminal', String(optional=True)),
        ('source_utm_campaign', String(optional=True)),
        ('source_utm_content', String(optional=True)),
        ('source_utm_medium', String(optional=True)),
        ('source_utm_source', String(optional=True)),
        ('source_utm_term', String(optional=True)),
        ('source_partner', String(optional=True)),
        ('source_subpartner', String(optional=True)),
        ('source_partner_uid', String(optional=True)),
        ('source_test_id', String(optional=True)),
        ('source_test_buckets', String(optional=True)),
        ('source_icookie', String(optional=True)),
        ('source_is_transfer', Boolean(optional=True)),
        ('source_serp_uuid', String(optional=True)),
        ('station_from_id', UInt64(optional=True, description='Id начальной станции маршрута пассажира')),
        ('station_to_id', UInt64(optional=True, description='Id конечной станции маршрута пассажира')),
        ('tickets_with_places_count', UInt64(description='Количество билетов с местами')),
        ('tickets_without_places_count', UInt64(description='Количество билетов без места')),
        ('total_fee_amount', Float(description='Сумма коммиссии при покупке')),
        ('total_insurance_amount', Float(description='Сумма страховок при покупке')),
        ('total_insurance_profit_amount', Float(description='Суммарная доля яндекса от страховок '
                                                            '(за вычетом возвратов)')),
        ('total_partner_fee_amount', Float(description='Суммарная коммиссия ИМ от продаж')),
        ('total_partner_refund_fee_amount', Float(description='Суммарная коммиссия ИМ от возвратов')),
        ('total_refund_fee_amount', Float(description='Сумма возвратов коммиссии')),
        ('total_refund_insurance_amount', Float(description='Сумма возвратов страховок (с учетом автовозврата)')),
        ('total_refund_ticket_amount', Float(description='Сумма возвратов билетов и услуг')),
        ('total_service_amount', Float(description='Сумма услуг')),
        ('total_tariff_amount', Float(description='Сумма билетов')),
        ('total_ticket_count', UInt64(description='Количество оформленных билетов')),
        # used for non-orchestrator orders only
        ('train_internal_status', String(description='Внутренний статус заказа, различается для im и im_boy')),
        ('train_name', String(optional=True, description='Название поезда')),
        ('train_number', String(description='Номер поезда')),
        ('train_ticket_number', String(description='Номер поезда trainTicketNumber')),
        ('two_storey', Boolean(description='Признак второго этажа')),
        ('uid', String(description='Внутренний Uid заказа')),
        ('user_ip', String(description='Ip пользователя')),
        ('user_is_mobile', Boolean(optional=True, description='Признак мобильной версии, приходит от фронтенда')),
        ('user_passport_uid', String(optional=True, description='passport_uid')),
        ('user_region_id', UInt64(description='geoID пользователя')),
        ('user_yandex_uid', String(optional=True, description='yandex_uid')),
        ('boy_order_id', String(optional=True, description='Идентификатор заказа BOY order_pretty_id (uid для im)')),
        ('item_state', Enumeration(ItemState, String, optional=True, description='Состояние услуги в заказе')),
    ])


class SuburbanOrder(Schematized):
    __fields__ = OrderedDict([
        ('provider', String(description='Поставщик данных')),
        ('provider_order_id', String(description='Id заказа у поставщика данных')),
        ('carrier_partner', String(description='Перевозчик')),
        ('station_from_id', UInt64(description='Id станции отправления в базе Расписаний')),
        ('station_to_id', UInt64(description='Id станции прибытия в базе Расписаний')),
        ('station_from_title', String(description='Название станции отправления')),
        ('station_to_title', String(description='Название станции прибытия')),
        ('departure_date', String(description='Дата, на которую куплен билет')),
    ])


class BusesOrder(Schematized):
    __fields__ = OrderedDict([
        ('active_ticket_count', UInt64(optional=True, description='Количество не возвращенных билетов')),
        ('adult_passengers_count', UInt64(description='Количество взрослых пассажиров')),
        ('arrival', UInt64(optional=True, description='Время прибытия, время местное')),
        ('baggage_tickets_count', UInt64(optional=True, description='Количество билетов багажа без учёта '
                                                                    'возвращённых')),
        ('billing_info', String(optional=True, description='Информация о движениях по заказу')),
        ('bus_internal_status', String(description='Автобусный статус заказа')),
        ('bus_model', String(optional=True, description='Модель, описание транспортного средства')),
        ('carrier_id', String(optional=True, description='Текстовый код перевозчика')),
        ('children_with_seats_count', UInt64(optional=True, description='Количество детских билетов')),
        ('departure', UInt64(description='Время отправления, время местное')),
        ('finished_at', UInt64(optional=True, description='Время завершения покупки, Timestamp UTC')),
        ('from_id', String(optional=True, description='Пункт отправления рейса, может быть пустой из-за отсутствия '
                                                      'матчинга')),
        ('from_partner_description', String(optional=True, description='Описание пункта отправления рейса')),
        ('online_refund', Boolean(description='Возможность онлайн возврата')),
        ('refunded_ticket_count', UInt64(description='Количество возвращенных билетов.')),
        ('requested_ticket_count', UInt64(description='Количество запрошенных для бронирования билетов')),
        ('route_name', String(optional=True, description='Название маршрута, если есть')),
        ('search_from_id', String(description='from-id по которому произведен поиск')),
        ('search_to_id', String(description='to-id по которому произведен поиск')),
        ('to_id', String(optional=True, description='Пункт прибытия, может быть пустой из-за отсутствия матчинга')),
        ('to_partner_description', String(optional=True, description='Описание пункта прибытия рейса')),
        ('total_agency_fee_amount', Float(optional=True, description='Сумма агентского вознаграждения по заказу')),
        ('total_fee_amount', Float(description='Сумма коммиссии при покупке')),
        ('total_partner_fee_amount', Float(description='Сбор партнера')),
        ('total_partner_refund_fee_amount', Float(description='	Возвраты сбора партнера')),
        ('total_refund_fee_amount', Float(description='Сумма возвратов коммиссии')),
        ('total_refund_ticket_amount', Float(description='Сумма возвратов билетов')),
        ('total_tariff_amount', Float(description='Сумма билетов')),
        ('total_ticket_count', UInt64(description='Количество оформленных билетов')),
        ('partner_commission', Float(optional=True, description='База для расчета агентского вознаграждения для '
                                                                'нестандартных схем, типа Басфора')),
        ('uid', String(description='Внутренний Uid заказа')),
        ('label_campaign', String(optional=True, description='')),
        ('label_content', String(optional=True, description='')),
        ('label_gclid', String(optional=True, description='Идентификатор перехода с рекламы в Гугле, сохраняется на '
                                                          'сессию')),
        ('label_icookie', String(optional=True, description='Новый идентификатор пользователя, замена yandexuid')),
        ('label_int_test_ids', String(optional=True, description='')),
        ('label_medium', String(optional=True, description='')),
        ('label_partner', String(optional=True, description='Переходы с ссылок партнёрской программы')),
        ('label_partner_uid', String(optional=True, description='')),
        ('label_passport_uid', String(optional=True, description='uid пользователя')),
        ('label_platform', String(optional=True, description='Платформа: веб, ios или android')),
        ('label_req_id', String(optional=True, description='ID запроса на СЕРПе Яндекса, передаётся в параметрах '
                                                           'ссылки при переходе с колдунщика. Сохраняется на сессию')),
        ('label_source', String(optional=True, description='')),
        ('label_subpartner', String(optional=True, description='')),
        ('label_term', String(optional=True, description='')),
        ('label_user_device', String(optional=True, description='Устройство пользователя')),
        ('label_yandex_uid', String(optional=True, description='yandex_uid')),
        ('label_ytp_referer', String(optional=True, description='Атрибуция трафика с HappyPage отелей')),

        ('provider', String(optional=True, description='Партнер для boy заказов')),
        ('boy_order_id', String(optional=True, description='Идентификатор заказа BOY order_pretty_id')),
        ('payment_ts', UInt64(optional=True, description='Время завершения оплаты, Timestamp UTC')),
    ])


class ToursOrder(Schematized):
    __fields__ = OrderedDict([
        ('partner_status', String(optional=True)),
        ('check_in', String(optional=True)),
        ('check_out', String(optional=True)),
        ('hotel_name', String(optional=True)),
        ('hotel_pansion', String(optional=True)),
        ('country_name', String(optional=True)),
        ('city_name', String(optional=True)),
        ('adults', Int64()),
        ('children', Int64()),
        ('infants', Int64()),
        ('fuel_charge', Float()),
    ])


class TrainOrderSnapshot(OrderSnapshot, TrainOrder):
    __fields__ = OrderedDict([
        ('category', String('train', description='train')),
        ('snapshot_source', String('collectors', optional=True))
    ])


class SuburbanOrderSnapshot(OrderSnapshot, SuburbanOrder):
    __fields__ = OrderedDict([
        ('category', String('suburban', description='suburban')),
        ('snapshot_source', String('collectors', optional=True)),
        ('boy_order_id', String(description='Идентификатор заказа BOY')),
    ])


class AeroflotAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('aeroflot')),
        ('source', String()),
    ])


class AeroflotClickoutAviaOrderSnapshot(AeroflotAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('source', String('aeroflot_clickout')),
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('trip_type', String(optional=True)),
    ])


class AgentAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('agent')),
    ])


class AviaPartnerBookingLogSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('trip_type', String(optional=True)),
        ('source_updated_at', UInt64()),
    ])


class AviakassaAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('aviakassa')),
        ('source', String('aviakassa')),
    ])


class AviaoperatorAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('aviaoperator')),
        ('source', String('aviaoperator')),
    ])


class AzimuthAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('azimuth')),
        ('source', String('azimuth')),
    ])


class BiletdvAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('biletdv')),
        ('source', String('biletdv')),
        ('trip_type', String(optional=True)),
        ('currency_code', Enumeration(OrderCurrencyCode, String, default=OrderCurrencyCode.RUB)),
    ])


class BiletikaeroagAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('biletikaeroag')),
        ('source', String('biletikaeroag')),
    ])


class BiletinetAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('biletinet')),
        ('source', String('biletinet')),
    ])


class BiletixAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('biletix')),
    ])


class BookandtripAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('booktripruag')),
        ('source', String('booktripruag')),
        ('trip_type', String(optional=True)),
        ('adults', UInt64(1)),
        ('children', UInt64(0)),
        ('infants', UInt64(0)),
    ])


class CitytravelAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('citytravel')),
        ('source', String('citytravel')),
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('trip_type', String(optional=True)),
    ])


class Citytravel1AviaOrderSnapshot(CitytravelAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('citytravel1')),
    ])


class ClickaviaAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('clickavia')),
        ('source', String('clickavia')),
    ])


class ExpressaviaAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('expressavia')),
    ])


class FlyoneAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('source', String('flyone')),
        ('partner_name', String('flyone')),
    ])


class GogateAviaOrderSnapshot(AviaPartnerBookingLogSnapshot):
    __fields__ = OrderedDict([
        ('source', String('gogate')),
        ('partner_name', String('gogate')),
    ])


class KiwiAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('kiwi')),
        ('source', String('kiwi')),
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('currency_code', Enumeration(OrderCurrencyCode, String, default=OrderCurrencyCode.EUR)),
    ])


class KupibiletAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('kupibilet')),
        ('source', String('kupibilet')),
        ('trip_type', String(optional=True)),
    ])


class MegotravelAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('megotravel')),
        ('source', String('megotravel')),
    ])


class NabortuAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('nabortu')),
        ('source', String('nabortu')),
        ('source_updated_at', UInt64()),
    ])


class NebotravelAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('nebotravel')),
        ('source', String('nebotravel')),
    ])


class UralAirlinesAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('uralairlines')),
        ('source', String('u6')),
    ])


class OnetwotripruAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('onetwotripru')),
    ])


class OzonAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ozon')),
    ])


class SmartaviaAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('smartavia')),
        ('source', String('smartavia')),
        ('partner_profit', Float(optional=True))
    ])


class TripcomHistoryAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('source', String('tripcom')),
        ('partner_name', String('tripcom')),
    ])


class TripcomAviaOrderSnapshot(AviaPartnerBookingLogSnapshot):
    __fields__ = OrderedDict([
        ('source', String('tripcom')),
        ('partner_name', String('tripcom')),
    ])


class UtairAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('utair')),
        ('source', String('utair')),
    ])


class AnywayanydayAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('anywayanyday')),
        ('source', String('anywayanyday')),
    ])


class PobedaAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('pobeda')),
        ('source', String('pobeda')),
        ('origin', String(optional=True)),
        ('destination', String(optional=True)),
        ('currency_code', Enumeration(OrderCurrencyCode, String, default=OrderCurrencyCode.RUB)),
    ])


class RedwingsAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('redwings')),
    ])


class RuslineAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('rusline')),
    ])


class SSevenAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('s_seven')),
        ('source', String('s_seven')),
        (
            'service_fee',
            Float(default=0., description='Сумма, за которую комиссию мы не получаем, входит в order_amount')
        ),
        ('partner_profit', Float(optional=True, description='Сумма нашей выручки по расчетам партнера')),
    ])


class SuperkassaAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('superkassa')),
        ('source', String('superkassa')),
    ])


class SupersaverAviaOrderSnapshot(AviaPartnerBookingLogSnapshot):
    __fields__ = OrderedDict([
        ('source', String('supersaver')),
        ('partner_name', String('supersaver')),
    ])


class SvyaznoyAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('svyaznoy')),
    ])


class TinkoffAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('tinkoff')),
        ('source', String('tinkoff')),
        ('trip_type', String(optional=True)),
        ('adults', UInt64(1)),
        ('children', UInt64(0)),
        ('infants', UInt64(0)),
    ])


class TutuAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('tutu')),
        ('source', String('tutu')),
        ('trip_type', String(optional=True)),
        ('adults', UInt64(1)),
        ('children', UInt64(0)),
        ('infants', UInt64(0)),
        ('unisearchquery', String(optional=True))
    ])


class TicketsruAviaOrderSnapshot(AviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ticketsru')),
        ('source', String('ticketsru')),
        ('trip_type', String(optional=True)),
        ('adults', UInt64(1)),
        ('children', UInt64(0)),
        ('infants', UInt64(0)),
    ])


class TripruAviaOrderSnapshot(AviaPartnerBookingLogSnapshot):
    __fields__ = OrderedDict([
        ('source', String('trip_ru')),
        ('partner_name', String('trip_ru')),
    ])


class UzairwaysAviaOrderSnapshot(NemoAviaOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('uzairways')),
    ])


class BookingOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('booking')),
        ('hotel_type', String(optional=True, empty_is_none=True)),
    ])


class Hotels101OrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('hotels101')),
    ])


class HotelsCombinedOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('hotelscombined')),
        ('hotel_name', String(optional=True)),
    ])


class LevelTravelOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('leveltravel')),
    ])


class OstrovokOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ostrovok')),
    ])


class BronevikOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('bronevik')),
        ('permalink', String()),
        ('profit_amount', Float(optional=True)),
    ])


class ExpediaOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('expedia')),
        ('permalink', String()),
        ('itinerary_id', String()),
        ('confirmation_id', String()),
        ('profit_amount', Float(optional=True))
    ])


class DolphinOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('dolphin')),
        ('permalink', String()),
        ('code', String()),
        ('profit_amount', Float(optional=True)),
    ])


class TravellineOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('travelline')),
        ('permalink', String()),
        ('confirmation_id', String()),
        ('profit_amount', Float()),
    ])


class BnovoOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('bnovo')),
        ('permalink', String()),
        ('confirmation_id', String()),
        ('profit_amount', Float()),
    ])


class TvilOrderSnapshot(HotelsOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('tvil')),
        ('profit_amount', Float(optional=True)),
    ])


class ImTrainOrderSnapshot(TrainOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('im', description='im')),
    ])


class ImBoyTrainOrderSnapshot(TrainOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('im_boy', description='im_boy')),
    ])


class SuburbanBoySuburbanOrderSnapshot(SuburbanOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('suburban', description='suburban')),
    ])


class BusesOrderSnapshot(OrderSnapshot, BusesOrder):
    __fields__ = OrderedDict([
        ('category', String('buses')),
        ('snapshot_source', String('collectors', optional=True)),
    ])


class BoyBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('bus-boy')),
    ])


class AtlasbusBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('atlasbus')),
    ])


class BusforBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('busfor')),
    ])


class EcolinesBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ecolines')),
    ])


class EtrafficBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('etraffic')),
    ])


class NoyBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('noy')),
    ])


class OkBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ok')),
    ])


class RusetBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('ruset')),
    ])


class SksBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('sks')),
    ])


class UnitikiNewBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('unitiki-new')),
    ])


class YugavtotransBusesOrderSnapshot(BusesOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('yugavtotrans')),
    ])


class ToursOrderSnapshot(OrderSnapshot, ToursOrder):
    __fields__ = OrderedDict([
        ('category', String('tours')),
        ('snapshot_source', String('collectors', optional=True)),
    ])


class LevelTravelWhitelabelOrderSnapshot(ToursOrderSnapshot):
    __fields__ = OrderedDict([
        ('partner_name', String('leveltravel_whitelabel')),
    ])


PROVIDERS = [
    # Avia
    AeroflotClickoutAviaOrderSnapshot,
    AgentAviaOrderSnapshot,
    AnywayanydayAviaOrderSnapshot,
    AviakassaAviaOrderSnapshot,
    AviaoperatorAviaOrderSnapshot,
    AzimuthAviaOrderSnapshot,
    BiletdvAviaOrderSnapshot,
    BiletikaeroagAviaOrderSnapshot,
    BiletinetAviaOrderSnapshot,
    BiletixAviaOrderSnapshot,
    BookandtripAviaOrderSnapshot,
    CitytravelAviaOrderSnapshot,
    Citytravel1AviaOrderSnapshot,
    ClickaviaAviaOrderSnapshot,
    ExpressaviaAviaOrderSnapshot,
    GogateAviaOrderSnapshot,
    KiwiAviaOrderSnapshot,
    KupibiletAviaOrderSnapshot,
    MegotravelAviaOrderSnapshot,
    NabortuAviaOrderSnapshot,
    NebotravelAviaOrderSnapshot,
    UralAirlinesAviaOrderSnapshot,
    OzonAviaOrderSnapshot,
    PobedaAviaOrderSnapshot,
    RedwingsAviaOrderSnapshot,
    RuslineAviaOrderSnapshot,
    SmartaviaAviaOrderSnapshot,
    SSevenAviaOrderSnapshot,
    SuperkassaAviaOrderSnapshot,
    SupersaverAviaOrderSnapshot,
    SvyaznoyAviaOrderSnapshot,
    TinkoffAviaOrderSnapshot,
    TicketsruAviaOrderSnapshot,
    TripruAviaOrderSnapshot,
    TutuAviaOrderSnapshot,
    UzairwaysAviaOrderSnapshot,
    UtairAviaOrderSnapshot,
    TripcomAviaOrderSnapshot,
    # Hotels
    BnovoOrderSnapshot,
    BronevikOrderSnapshot,
    BookingOrderSnapshot,
    DolphinOrderSnapshot,
    ExpediaOrderSnapshot,
    Hotels101OrderSnapshot,
    HotelsCombinedOrderSnapshot,
    LevelTravelOrderSnapshot,
    OstrovokOrderSnapshot,
    TravellineOrderSnapshot,
    TvilOrderSnapshot,
    # Train
    ImTrainOrderSnapshot,
    ImBoyTrainOrderSnapshot,
    # Suburban
    SuburbanBoySuburbanOrderSnapshot,
    # Buses
    AtlasbusBusesOrderSnapshot,
    BusforBusesOrderSnapshot,
    EcolinesBusesOrderSnapshot,
    EtrafficBusesOrderSnapshot,
    NoyBusesOrderSnapshot,
    OkBusesOrderSnapshot,
    RusetBusesOrderSnapshot,
    SksBusesOrderSnapshot,
    UnitikiNewBusesOrderSnapshot,
    YugavtotransBusesOrderSnapshot,
    BoyBusesOrderSnapshot,
    # Tours
    LevelTravelWhitelabelOrderSnapshot,
    # generic
    GenericOrderSnapshot,
]
