# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import namedtuple
from textwrap import dedent
import os

from jinja2 import Template
from six.moves.urllib.parse import urlparse, urlunparse

from travel.cpa.collectors.lib.http_collector import HttpCollector
from travel.cpa.collectors.nomenclature import COLLECTORS


CollectorInfo = namedtuple(
    'CollectorInfo',
    [
        'partner_name',
        'host',
        'set_host_header',
        'include_x_forwarded_for',
        'include_x_forwarded_for_as_remote_addr',
    ]
)


TEMPLATE_TEXT = '''
    # this file was generated by travel/cpa/tools/build_proxy_config.py
    # do not edit it manually

    server {
        include listen;

        location /ping {
            proxy_pass  http://localhost:${PROBES_SERVER_PORT}/ping;
        }

        location /shutdown {
            proxy_pass  http://localhost:${PROBES_SERVER_PORT}/shutdown;
        }

        {% for collector in collectors %}
        location /{{ collector.partner_name }}/ {
            # Don't pass x-* headers
            {% if not collector.include_x_forwarded_for is sameas true -%}
            proxy_set_header x-forwarded-for "";
            {% elif collector.include_x_forwarded_for_as_remote_addr is sameas true -%}
            proxy_set_header x-forwarded-for $remote_addr;
            {% endif -%}
            proxy_set_header x-qloud-ssl-verified "";
            proxy_set_header x-forwarded-host "";
            proxy_set_header x-qloud-environment "";
            proxy_set_header x-original-url "";
            proxy_set_header x-balancer-host "";
            proxy_set_header x-forwarded-proto "";
            proxy_set_header x-forwarded-for-y "";
            proxy_set_header x-req-id "";
            proxy_set_header x-real-ip "";
            proxy_set_header x-request-id  "";
            {% if collector.set_host_header is sameas true %}
            proxy_set_header Host $host;
            {% endif %}
            proxy_send_timeout 181s;
            proxy_read_timeout 181s;

            proxy_pass {{ collector.host }}/;
        }
        {% endfor %}
    }

'''


def get_url_host(url):
    url_parts = urlparse(url)
    return urlunparse(url_parts[:2] + 4 * ('',))


def run(args):
    collectors = list()
    for collector in COLLECTORS:
        if not issubclass(collector, HttpCollector):
            continue
        if not collector.ALLOW_PROXY:
            continue
        if collector.BASE_URL is not None:
            collector_info = CollectorInfo(
                collector.PARTNER_NAME,
                get_url_host(collector.BASE_URL),
                collector.SET_HOST_HEADER,
                False,
                False,
            )
            collectors.append(collector_info)
        else:
            print('WARNING: Value for "BASE_URL" not defined for class {}'.format(collector))
        if collector.BASE_URL_TESTING is not None:
            collector_info = CollectorInfo(
                collector.PARTNER_NAME + "_testing",
                get_url_host(collector.BASE_URL_TESTING),
                collector.SET_HOST_HEADER,
                False,
                False,
            )
            collectors.append(collector_info)

    collectors.append(CollectorInfo('boy_aeroflot', 'https://gw.aeroflot.ru', False, True, True))
    collectors.append(CollectorInfo('boy_aeroflot_testing', 'https://gw.aeroflot.ru', False, True, True))

    collectors.append(CollectorInfo('variflight', 'http://open-al.variflight.com', False, True, False))

    collectors.append(CollectorInfo("travelline", "https://yandex.tlintegration.com", False, True, False))
    collectors.append(CollectorInfo("bnovo_public", "http://public-api.reservationsteps.ru", False, True, False))
    collectors.append(CollectorInfo("bnovo_private", "http://api.reservationsteps.ru", False, True, False))

    collectors.append(CollectorInfo("tvil_testing_static", "https://kakugodno.tvil.ru", False, True, False))
    collectors.append(CollectorInfo("tvil_prod_static", "https://tvil.ru", False, True, False))

    template = Template(TEMPLATE_TEXT)
    config = template.render(collectors=collectors)
    config = dedent(config)
    fn = os.path.expanduser(args.output_fn)
    with open(fn, 'w') as f:
        f.write(config)
