# -*- coding: utf-8 -*-

from typing import Any, Dict, List
import logging

from yt.wrapper import YtClient
import yaml
import yt.wrapper as yt

from travel.cpa.data_processing.lib.order_data_model import CATEGORY_CONFIGS
from travel.cpa.lib.lib_yt import get_table_schema
from travel.library.python.time_interval import TimeInterval
from travel.library.python.arcadia.arcadia_tools import get_arcadia_path


Schema = List[Dict[str, str]]


def create_table(yt_client: YtClient, root: str, table_config: Dict[str, Any], schema: Schema):
    table_path = yt.ypath_join(root, table_config['path'])
    logging.info(f'Processing {table_path}')

    attributes = table_config['attributes'].copy()
    attributes['schema'] = schema
    attributes['min_data_ttl'] = TimeInterval(attributes.get('min_data_ttl', 0)).interval * 1000
    attributes['max_data_ttl'] = TimeInterval(attributes.get('max_data_ttl', 0)).interval * 1000

    if yt_client.exists(table_path):
        logging.info('Table already exists')
        return

    logging.info('Creating table')
    yt_client.create('table', table_path, recursive=True, attributes=attributes)
    logging.info('Mounting table')
    yt_client.mount_table(table_path)


def create_dump_links(yt_client, root, category):
    parent = yt.ypath_dirname(root)
    link_path = yt.ypath_join(parent, 'cpa', category, 'orders')
    table_path = yt.ypath_join(parent, 'cpa_public', category, 'orders')
    if yt_client.exists(table_path):
        logging.info(f'Linking from {link_path} to {table_path}')
        yt_client.link(table_path, link_path, recursive=True, force=True)


def run(args):
    config_path = get_arcadia_path('travel', 'cpa', 'tools3', 'tables_config.yaml')
    with open(config_path) as f:
        config = yaml.load(f.read(), yaml.SafeLoader)

    root = config['roots'][args.env]
    root = root.format(user=args.user)
    logging.info(f'Root: {root}')

    yt_token = args.yt_token
    yt_clients = [YtClient(proxy=p, token=yt_token) for p in args.yt_proxies]

    categories = config['categories']
    order_table_config = config['order_tables']

    for yt_client in yt_clients:
        for table_config in config['tables']:
            create_table(yt_client, root, table_config, table_config['schema'])

        for category in categories:
            category_config = CATEGORY_CONFIGS[category]
            order_cls = category_config.order_with_decoded_label_cls or category_config.order_with_encoded_label_cls
            schema = order_cls().get_yt_schema()
            schema = get_table_schema(schema, sort_by=['partner_name', 'partner_order_id'])

            category_path = yt.ypath_join(root, category)
            create_table(yt_client, category_path, order_table_config, schema)

            create_dump_links(yt_client, root, category)
