#!/usr/bin/env python3
# encoding: utf-8

import os
import logging
import logging.config
import argparse

from travel.devops.solomon.alerts import SolomonAlerts
from travel.devops.solomon.client import SolomonClient
from travel.devops.solomon.tool import SolomonTool
from travel.devops.solomon.utils import get_default_project_path, build_arcadia_path, load_config

LOG_CONFIG = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'stdout': {
            'class': 'logging.StreamHandler',
            'stream': 'ext://sys.stdout',
        },
    },
    'loggers': {
        '': {
            'handlers': ['stdout'],
            'level': 'INFO',
        },
    },
}

LOG = logging.getLogger(__name__)


def main():
    parser = argparse.ArgumentParser(description='A tool to deal with Solomon objects')

    parser.add_argument('--auth-token-path', default=os.path.expanduser('~/.solomon/token'))
    parser.add_argument('--auth-token-env', default='SOLOMON_TOKEN')
    parser.add_argument('--auth-token')

    parser.add_argument('--configs-path',
                        help='Path to solomon configs relative from the arcadia root, for ex. travel/hotels/devops/solomon',
                        default=get_default_project_path())

    parser.add_argument('--verbose', '-v', action='count', default=0, help='Increase verbosity')
    parser.add_argument('-n', '--dry-run', action='store_true', help='Perform a trial run with no changes made')

    subparsers = parser.add_subparsers(title='Commands to deal with local file database and corresponding remote Solomon objects')
    subparsers.required = True
    subparsers.dest = 'object_type'
    for object_type in ['graphs', 'dashboards']:
        subparser = subparsers.add_parser(object_type)
        subparser.add_argument('--backup', action='store_true', help=f'Backup files of {object_type} before updating them')
        subparser.add_argument('--service', dest='service_ids', action='append', default=[], help=f'Process only {object_type} related to specified services')
        subparser.add_argument('command', choices=['push', 'pull'], help=f'Commands for {object_type}')
        subparser.add_argument('object_ids', nargs='*', default=[], help=f'ids of {object_type} to change')

    subparser = subparsers.add_parser('alerts')
    subparser.add_argument('command', choices=['push'], help='Commands for alerts')
    subparser.add_argument('--env', action='append', default=[])
    subparser.add_argument('--alert', action='append', default=[])

    args = parser.parse_args()

    if not args.configs_path:
        print('Please specify path to configs. You could do this with:'
              '    --configs-path argument (for ex. --configs-path=travel/hotels/devops/solomon)'
              '    or write it to ~/.solomon/configs-path file (echo "travel/hotels/devops/solomon" > ~/.solomon/configs-path)'
              '    or by setting env variable TRAVEL_SOLOMON_CONFIGS_PATH=travel/hotels/devops/solomon')
        exit(1)

    loggers = LOG_CONFIG['loggers']
    if args.verbose == 1:
        loggers['']['level'] = 'DEBUG'
    elif args.verbose == 2:
        for k, v in loggers.items():
            v['level'] = 'DEBUG'
    elif args.verbose >= 3:
        for k, v in loggers.items():
            v['level'] = 'NOTSET'
    logging.config.dictConfig(LOG_CONFIG)

    root_path = build_arcadia_path(args.configs_path)
    project_config = load_config(root_path)
    solomon_client = SolomonClient(args, project_config.solomon_project)
    if args.object_type == 'alerts':
        SolomonAlerts(args, root_path, solomon_client).run()
    elif args.object_type in ['dashboards', 'graphs']:
        SolomonTool(args, root_path, solomon_client, project_config.solomon_project).run()
    else:
        raise NotImplementedError(f'{args.object_type} are not supported')


if __name__ == '__main__':
    main()
