# encoding: utf-8

import os
import logging

import requests
import difflib
from color import colored
import yaml


ENDPOINT = 'https://solomon.yandex-team.ru'

LOG = logging.getLogger(__name__)


class SolomonClient:
    def __init__(self, args, project):
        token = None
        if args.auth_token is not None:
            token = args.auth_token
        if token is None and args.auth_token_env is not None:
            token = os.environ.get(args.auth_token_env)
        if token is None and args.auth_token_path is not None:
            if os.path.exists(args.auth_token_path):
                with open(args.auth_token_path) as handle:
                    token = handle.readline().strip()
        if token is None:
            raise ValueError("Missing Solomon token")
        self.session = requests.session()
        self.session.headers['Authorization'] = f'OAuth {token}'
        self.url_base = f'{ENDPOINT}/api/v2/projects/{project}/'

        self.args = args

    @staticmethod
    def print_and_raise_for_status(response):
        if response.status_code != 200:
            LOG.error("%s", response.content)
            raise requests.HTTPError(f"Non-200 status code returned ({response.status_code})")

    def url_for(self, *args):
        return self.url_base + '/'.join(map(str, args))

    def read_all(self):
        response = self.session.get(self.url_for(self.args.object_type), params={'pageSize': 10000})
        self.print_and_raise_for_status(response)
        data = response.json()
        assert self.args.object_type == 'alerts' or data['page']['pagesCount'] == 1
        return data

    def read(self, object_id):
        response = self.session.get(self.url_for(self.args.object_type, object_id))
        self.print_and_raise_for_status(response)
        return response.json()

    def create(self, data, object_id):
        if self.args.dry_run:
            return
        data['id'] = object_id
        response = self.session.post(self.url_for(self.args.object_type), json=data)
        self.print_and_raise_for_status(response)
        return response.json()

    def update(self, data, object_id):
        if self.args.dry_run:
            return
        response = self.session.put(self.url_for(self.args.object_type, object_id), json=data)
        self.print_and_raise_for_status(response)
        return response.json()

    @staticmethod
    def drop_system_fields(data):
        system_fields = [
            'id',
            'projectId',
            'createdAt',
            'createdBy',
            'updatedAt',
            'updatedBy',
            'version',
            'generatedId',
        ]
        for key in system_fields:
            data.pop(key, None)

    @staticmethod
    def diff_data(old_data, new_data, object_id):
        old_lines = yaml.dump(old_data, sort_keys=True, allow_unicode=True).splitlines()
        new_lines = yaml.dump(new_data, sort_keys=True, allow_unicode=True).splitlines()
        for line in difflib.unified_diff(old_lines, new_lines, fromfile=object_id, tofile=object_id, lineterm=''):
            if line:
                if line[0] == '@':
                    line = colored(line, 'cyan')
                elif line[0] == '-':
                    line = colored(line, 'red')
                elif line[0] == '+':
                    line = colored(line, 'green')
            LOG.info("%s", line)
