# encoding: utf-8

import os
import logging

from color import colored
import yaml
from urllib.parse import urlparse, parse_qsl

EXTENSION = '.yaml'

LOG = logging.getLogger(__name__)


class ObjectTypes:
    GRAPHS = 'graphs'
    DASHBOARDS = 'dashboards'


class SolomonTool:
    def __init__(self, args, root_path, solomon_client, project):
        self.args = args
        self.project = project
        self.root_path = root_path
        self.object_root = os.path.join(self.root_path, self.args.object_type)
        with open(os.path.join(self.root_path, 'defaults.yaml')) as f:
            defaults = yaml.load(f, Loader=yaml.FullLoader)
        self.graph_defaults = defaults['graph']
        self.graph_element_defaults = defaults['graph_element']
        self.dashboard_defaults = defaults['dashboard']
        self.dashboard_panel_defaults = defaults['dashboard_panel']

        self.client = solomon_client

        self.remote_object_ids = self.get_object_ids()
        self.object_filepaths = self.get_object_filepaths(self.object_root)

        self._all_graphs = None

        LOG.debug(f"there are {len(self.remote_object_ids)} {self.args.object_type} on endpoint and {len(self.object_filepaths)} on local file database")

    @staticmethod
    def get_object_filepaths(object_root):
        object_filepaths = dict()
        for dirpath, dirnames, filenames in os.walk(object_root):
            for filename in filenames:
                if not filename.endswith(EXTENSION):
                    continue
                object_id = filename[:-len(EXTENSION)]
                object_filepath = os.path.join(dirpath, filename)
                if object_id in object_filepaths:
                    LOG.error(f"there are at least two files for object {object_id}: {object_filepaths[object_id]} and {object_filepath}")
                    raise ValueError("Duplicated filenames in local file database")
                object_filepaths[object_id] = object_filepath
        return object_filepaths

    @staticmethod
    def namevalue_to_dict(data, key):
        mappings = dict()
        if key in data:
            for mapping in data[key]:
                mappings[mapping['name']] = mapping['value']
            data[key] = mappings
        return mappings

    @staticmethod
    def dict_from_namevalue(data, key):
        if key in data:
            mappings = []
            for name, value in data[key].items():
                mappings.append({'name': name, 'value': value})
            data[key] = mappings

    @staticmethod
    def add_defaults(data, defaults):
        for name, value in defaults.items():
            if type(value) is dict:
                for k, v in value.items():
                    if k not in data[name]:
                        data[name][k] = v
            elif name not in data:
                data[name] = value

    @staticmethod
    def delete_defaults(data, defaults):
        for key, value in defaults.items():
            if type(value) is dict:
                SolomonTool.namevalue_to_dict(data, key)
                curr = data.get(key, dict())
                for k, v in value.items():
                    if curr.get(k) == v:
                        curr.pop(k)
                SolomonTool.dict_from_namevalue(data, key)
            elif data.get(key) == value:
                del data[key]

    def normalize_graph_from_yaml(self, data):
        pass

    def normalize_dashboard_from_yaml(self, data):
        self._ensure_all_graphs_loaded()

        dashboard_parameters = data['parameters']
        for row in data['rows']:
            for panel in row['panels']:
                if 'graph' in panel:
                    selectors = panel.pop('selectors', {})
                    graph = panel['graph']
                    graph_params = self._all_graphs.get(graph, {}).get('parameters', {})
                    for k, v in graph_params.items():
                        if k not in selectors and k in dashboard_parameters and v != dashboard_parameters[k]:
                            selectors[k] = v
                        if k not in selectors:
                            if v == '*':
                                selectors[k] = '{{%s}}' % k
                            else:
                                selectors[k] = v
                    if selectors:
                        panel['selectors'] = selectors
                    if 'title' not in panel and graph in self._all_graphs:
                        panel['title'] = self._all_graphs[graph]['name']

    def convert_graph_from_solomon(self, data):
        self.delete_defaults(data, self.graph_defaults)
        self.namevalue_to_dict(data, 'parameters')
        for element in data['elements']:
            element_type = element.pop('type')
            if element_type == 'SELECTORS':
                self.namevalue_to_dict(element, 'selectors')
            elif element_type == 'EXPRESSION':
                pass
            else:
                raise NotImplementedError("unknown element type")
            self.delete_defaults(element, self.graph_element_defaults)

    def convert_graph_to_solomon(self, data):
        self.add_defaults(data, self.graph_defaults)
        self.dict_from_namevalue(data, 'parameters')
        for element in data['elements']:
            if 'expression' in element:
                assert 'selectors' not in element
                element['type'] = 'EXPRESSION'
            elif 'selectors' in element:
                assert 'expression' not in element
                element['type'] = 'SELECTORS'
                self.dict_from_namevalue(element, 'selectors')
            else:
                raise NotImplementedError("cannot determine element type")

    def get_object_ids(self):
        data = self.client.read_all()
        if not data:
            LOG.info(f"{self.args.object_type}: there are no resources of this type")
            return []
        object_ids = []
        for datum in data['result']:
            object_ids.append(datum['id'])
        return object_ids

    def get_objects(self, object_ids=None):
        if object_ids is None:
            object_ids = []
        if object_ids:
            missing_object_ids = set(object_ids) - set(self.remote_object_ids)
            if missing_object_ids:
                LOG.error("these object ids are missing on endpoint side: %s", ", ".join(missing_object_ids))
                raise ValueError("Missing object ids")
        else:
            object_ids = self.remote_object_ids
        LOG.debug(f"{self.args.object_type}: there are {len(object_ids)} objects to read from endpoint")

        objects = dict()
        for object_id in object_ids:
            data = self.client.read(object_id)
            object_id = data.pop('id')
            if object_id in objects:
                LOG.warning(f"duplicated object id {object_id}")
                continue
            objects[object_id] = data
        return objects

    def parse_graph_url(self, url):
        attributes = urlparse(url)
        if attributes.scheme != '' or attributes.netloc != '' or attributes.path not in ['', '/'] or attributes.params != '' or attributes.fragment != '':
            return None, None
        if not attributes.query:
            return None, None
        selectors = dict()
        for name, value in parse_qsl(attributes.query):
            selectors[name] = value
        if 'dashboard' in selectors:
            return None, None
        graph = selectors.pop('graph')
        if graph == 'auto':
            return None, None
        if selectors.get('project') == self.project:
            del selectors['project']
        return graph, selectors

    def convert_dashboard_from_solomon(self, data):
        self.delete_defaults(data, self.dashboard_defaults)
        self.namevalue_to_dict(data, 'parameters')
        for row in data['rows']:
            for panel in row['panels']:
                self.delete_defaults(panel, self.dashboard_panel_defaults)
                url = panel.pop('url')
                graph, selectors = self.parse_graph_url(url)
                if graph:
                    panel['graph'] = graph
                    if selectors:
                        panel['selectors'] = selectors
                elif url:
                    panel['url'] = url

    def make_graph_url(self, graph, parameters):
        if graph == '':
            return ''
        url = []
        for name, value in parameters.items():
            url.append(f'{name}={value}')
        if not url:
            return ''
        if 'project' not in parameters:
            url.insert(0, f'project={self.project}')
        return '/?' + '&'.join(url) + f'&graph={graph}'

    def convert_dashboard_to_solomon(self, data):
        self._ensure_all_graphs_loaded()

        parameters = data.get('parameters', dict()).copy()
        self.add_defaults(data, self.dashboard_defaults)
        self.dict_from_namevalue(data, 'parameters')
        for row_idx, row in enumerate(data['rows']):
            for panel_idx, panel in enumerate(row['panels']):
                if not panel:
                    continue
                if 'graph' in panel:
                    assert 'url' not in panel
                    graph = panel.pop('graph')

                    auto_parameters = {name: f'{{{{{name}}}}}' for name in parameters}
                    auto_parameters.update(panel.pop('selectors', dict()))

                    if graph in self._all_graphs:
                        panel_parameters = self._all_graphs[graph]['parameters']
                        for key in panel_parameters.keys():
                            if key in auto_parameters:
                                panel_parameters[key] = auto_parameters[key]
                        if 'title' not in panel:
                            panel['title'] = self._all_graphs[graph]['name']

                        if 'service' not in panel_parameters:
                            raise Exception('"service" parameter must be defined for dashboard, panel or graph. '
                                            'Row idx %s, panel idx %s' % (row_idx, panel_idx))
                    elif auto_parameters['project'] != self.project:  # Allow graphs from other projects
                        panel_parameters = auto_parameters
                    else:
                        raise Exception("Unknown graph %s in row idx %s, panel idx %s" % (graph, row_idx, panel_idx))

                    panel['url'] = self.make_graph_url(graph, panel_parameters)

                self.add_defaults(panel, self.dashboard_panel_defaults)

    def normalize_data_from_yaml(self, data):
        if self.args.object_type == ObjectTypes.GRAPHS:
            self.normalize_graph_from_yaml(data)
        elif self.args.object_type == ObjectTypes.DASHBOARDS:
            self.normalize_dashboard_from_yaml(data)
        else:
            raise NotImplementedError(f"object type {self.args.object_type} is not yet supported")

    def convert_data_from_solomon(self, data):
        if self.args.object_type == ObjectTypes.GRAPHS:
            self.convert_graph_from_solomon(data)
        elif self.args.object_type == ObjectTypes.DASHBOARDS:
            self.convert_dashboard_from_solomon(data)
        else:
            raise NotImplementedError(f"object type {self.args.object_type} is not yet supported")

    def convert_data_to_solomon(self, object_id, data):
        try:
            if self.args.object_type == ObjectTypes.GRAPHS:
                self.convert_graph_to_solomon(data)
            elif self.args.object_type == ObjectTypes.DASHBOARDS:
                self.convert_dashboard_to_solomon(data)
            else:
                raise NotImplementedError(f"object type {self.args.object_type} is not yet supported")
        except Exception:
            LOG.error("Failed to convert object '%s'" % object_id)
            raise

    def is_service_id_allowed(self, data):
        if self.args.service_ids:
            service_id = None
            if 'parameters' in data:
                service_id = data['parameters'].get('service')
            if service_id not in self.args.service_ids:
                return False
        return True

    def pull(self):
        created_count = 0
        updated_count = 0
        skipped_count = 0

        objects = self.get_objects(self.args.object_ids)
        for object_id, data in objects.items():
            self.client.drop_system_fields(data)
            self.convert_data_from_solomon(data)
            if not self.is_service_id_allowed(data):
                continue

            file_exists = object_id in self.object_filepaths

            if file_exists:
                object_filepath = self.object_filepaths[object_id]
            else:
                object_filepath = os.path.join(self.object_root, object_id + EXTENSION)

            if file_exists:
                with open(object_filepath) as f:
                    local_data = yaml.load(f, Loader=yaml.FullLoader)
                if local_data == data:
                    LOG.info(f"{object_id} skipped due to local and remote counterparts are identical")
                    skipped_count += 1
                    continue
                self.client.diff_data(local_data, data, object_id)
                updated_count += 1
            else:
                created_count += 1

            if not self.args.dry_run:
                if self.args.backup and file_exists:
                    os.rename(object_filepath, object_filepath + '~')
                with open(object_filepath, "w") as f:
                    yaml.dump(data, f, sort_keys=False, allow_unicode=True)

            LOG.info(f"{object_id}: file {object_filepath} %s", colored("updated", 'cyan') if file_exists else colored("created", 'green'))

        LOG.info(f"{created_count} {self.args.object_type} created, {updated_count} updated and {skipped_count} skipped")

    def push(self):
        for object_id in self.args.object_ids:
            if object_id not in self.object_filepaths:
                LOG.error(f"{object_id} file not found.\nAvailable files (first 10): {list(self.object_filepaths)[:10]}")
                return

        created_count = 0
        updated_count = 0
        skipped_count = 0

        for object_id, object_filepath in self.object_filepaths.items():
            if self.args.object_ids and object_id not in self.args.object_ids:
                continue
            LOG.info(f"Processing object {object_id}")

            with open(object_filepath) as f:
                data = yaml.load(f, Loader=yaml.FullLoader)
            if not self.is_service_id_allowed(data):
                continue
            self.normalize_data_from_yaml(data)

            if object_id in self.remote_object_ids:
                old_data = self.client.read(object_id)
                version = old_data.get('version')
                self.client.drop_system_fields(old_data)
                self.convert_data_from_solomon(old_data)
                if old_data == data:
                    LOG.info(f"{object_id} update skipped due to it has not changed")
                    skipped_count += 1
                    continue
                self.client.diff_data(old_data, data, object_id)
                self.convert_data_to_solomon(object_id, data)
                data['version'] = version
                self.client.update(data, object_id)
                LOG.info(f"{object_id} %s", colored("updated", 'cyan'))
                updated_count += 1
            else:
                self.convert_data_to_solomon(object_id, data)
                self.client.create(data, object_id)
                LOG.info(f"{object_id} %s", colored("created", 'green'))
                created_count += 1
            LOG.info(f"{object_id} from file {object_filepath} synchronized with remote counterpart")

        LOG.info(f"{created_count} {self.args.object_type} created, {updated_count} updated and {skipped_count} skipped")

    def run(self):
        assert self.args.object_type in [ObjectTypes.GRAPHS, ObjectTypes.DASHBOARDS]
        getattr(self, self.args.command)()

    def _ensure_all_graphs_loaded(self):
        if self._all_graphs is None:
            self._all_graphs = self._load_all_graphs()

    def _load_all_graphs(self):
        graphs = dict()
        for object_id, object_filepath in self.get_object_filepaths(os.path.join(self.root_path, ObjectTypes.GRAPHS)).items():
            with open(object_filepath) as f:
                graphs[object_id] = yaml.load(f, Loader=yaml.FullLoader)
        return graphs
