#!/usr/bin/env python3

import os
import yaml
import logging
from collections import OrderedDict


def patch(dst, src):
    if src is not None:
        for key in ("title", "rank"):
            if key in src:
                dst[key] = src[key]


def extract_toc_from_file(file_path, root_path):
    partial_toc = OrderedDict({
        "title": "[" + os.path.basename(file_path) + "]",
        "rank": 100,
        "href": file_path[len(root_path)+1:],
    })
    with open(file_path) as handle:
        data = handle.readlines()
        try:
            begin_idx = data.index("---\n") + 1
            end_idx = data.index("---\n", begin_idx)
            header = yaml.safe_load("".join(data[begin_idx:end_idx]))
        except ValueError:
            logging.info(f"File {file_path} does not contain YAML metadata header.")
            header = None
        patch(partial_toc, header)
    return partial_toc


def extract_toc_from_dir(dir_path, root_path):
    partial_toc = OrderedDict({
        "title": "[" + os.path.basename(dir_path) + "]",
        "rank": 100,
        "items": []
    })
    items = []
    header_path = os.path.join(dir_path, ".toc")
    if os.path.exists(header_path):
        with open(header_path) as handle:
            header = yaml.safe_load(handle)
            if header is not None:
                if "title" in header:
                    partial_toc["title"] = header["title"]
                if "rank" in header:
                    partial_toc["rank"] = header["rank"]
                if "include" in header:
                    items = header["include"]
    if len(items) == 0:
        for item in os.listdir(dir_path):
            item_path = os.path.join(dir_path, item)
            if os.path.isdir(item_path) and not item.startswith("_"):
                items.append(item)
            elif item.endswith(".md"):
                if item == "index.md":
                    partial_toc["href"] = item_path[len(root_path)+1:]
                    continue
                items.append(item)
    def item_mapper(item):
        item_path = os.path.join(dir_path, item)
        if os.path.isfile(item_path):
            return extract_toc_from_file(item_path, root_path)
        if os.path.isdir(item_path):
            return extract_toc_from_dir(item_path, root_path)
        return None
    partial_toc["items"] = list(map(item_mapper, items))
    return partial_toc


def finalize(toc, root=True):
    if not root:
        toc["name"] = toc["title"]
        del toc["title"]
        toc.move_to_end("name", last=False)
    del toc["rank"]
    items = toc.get("items", [])
    if len(items) > 0:
        items.sort(key=lambda item: (item.get("rank", 100), item.get("title", "")))
        for item in items:
            finalize(item, False)
        toc["items"] = items
    else:
        if "items" in toc:
            del toc["items"]


def main():
    root_path = os.path.dirname(os.path.realpath(__file__))

    toc = extract_toc_from_dir(root_path, root_path)
    finalize(toc)

    represent_dict_order = lambda self, data: self.represent_mapping("tag:yaml.org,2002:map", data.items())
    yaml.add_representer(OrderedDict, represent_dict_order)
    print(yaml.dump(toc, allow_unicode=True, default_flow_style=False))


if __name__ == "__main__":
    main()
