package ru.yandex.travel.externalapi.configuration;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.externalapi.service.providers.AdministratorInterfaceProvider;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.orders.proto.EServerState;
import ru.yandex.travel.orders.proto.HADiscoveryInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TPingRpcReq;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
@RequiredArgsConstructor
@Configuration
@EnableConfigurationProperties(HotelsAdministratorConfigurationProperties.class)
public class HotelsAdministratorConfiguration {

    private final HotelsAdministratorConfigurationProperties properties;

    @Bean("HotelsAdministratorSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean
    public TvmHeaderClientHelper getTvmHeaderClientHelper(@Autowired(required = false) TvmWrapper tvm) {
        if (tvm != null && properties.getTvm().isEnabled()) {
            return new TvmHeaderClientHelper(tvm);
        }
        return null;
    }

    @Bean
    public HAGrpcChannelFactory hotelsAdministratorChannelFactory(
            @Qualifier("HotelsAdministratorSupplier") ChannelSupplier channelSupplier,
            @Autowired(required = false) TvmHeaderClientHelper tvmHeaderClientHelper) {
        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder.withPingProducer(
                channel -> FutureUtils.buildCompletableFuture(HADiscoveryInterfaceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getState() == EServerState.SS_MASTER ? ChannelState.READY_MASTER :
                        ChannelState.READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(label -> this.createChannel(label, tvmHeaderClientHelper))
                .build();
    }

    @Bean
    public AdministratorInterfaceProvider administratorInterfaceProvider(
            @Qualifier("hotelsAdministratorChannelFactory") HAGrpcChannelFactory channelFactory) {
        return new AdministratorInterfaceProvider(channelFactory);
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(HADiscoveryInterfaceV1Grpc.getPingMethod().getFullMethodName())
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(Arrays.asList(
                loggingClientInterceptor,
                defaultTimeoutClientInterceptor));
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }

}
