package ru.yandex.travel.externalapi.configuration;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.config.MeterFilter;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import org.springframework.boot.actuate.autoconfigure.metrics.MeterRegistryCustomizer;
import org.springframework.boot.actuate.metrics.web.servlet.WebMvcTags;
import org.springframework.boot.actuate.metrics.web.servlet.WebMvcTagsProvider;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.metrics.MetricsUtils;

@Configuration
public class MeterConfiguration {
    private static final Duration[] SLAS = {
            Duration.ofMillis(10), Duration.ofMillis(50), Duration.ofMillis(100), Duration.ofMillis(150),
            Duration.ofMillis(200), Duration.ofMillis(500), Duration.ofMillis(1000), Duration.ofMillis(10000)
    };

    private static Tag tagFromStatus(HttpServletResponse response) {
        String status;
        if (response == null) {
            status = "UNKNOWN";
        } else if (Set.of(400, 401, 403).contains(response.getStatus())) {
            status = Integer.toString(response.getStatus());
        } else {
            status = response.getStatus() / 100 + "xx";
        }
        return Tag.of("status", status);
    }

    @Bean
    public MeterRegistryCustomizer<MeterRegistry> getRegistryCustomizer() {
        return registry -> {
            registry.config().meterFilter(new MeterFilter() {
                @Override
                public DistributionStatisticConfig configure(Meter.Id id, DistributionStatisticConfig config) {
                    if (id.getName().startsWith("http.server.requests")) {
                        return DistributionStatisticConfig.builder()
                                .percentilesHistogram(true)
                                .serviceLevelObjectives(Arrays.stream(SLAS).mapToDouble(Duration::toNanos).toArray())
                                .percentiles(MetricsUtils.higherPercentiles())
                                .build();
                    } else {
                        return config;
                    }
                }
            });
            AtomicBoolean initAllTagsInProgressFlag = new AtomicBoolean(false);
            registry.config().onMeterAdded(meter -> {
                if (meter.getId().getName().startsWith("http.server.requests") && (meter instanceof Timer)) {
                    if (!initAllTagsInProgressFlag.compareAndSet(false, true)) {
                        // some (probably) recursive call is already in progress
                        // the latest micrometer lib version makes the logic below trigger itself infinitely
                        return;
                    }
                    List<Tag> tags = meter.getId().getTags().stream().filter(tag -> !tag.getKey().equals("status")).collect(Collectors.toList());
                    for (Tag tag : Arrays.asList(
                            Tag.of("status", "2xx"), Tag.of("status", "3xx"), Tag.of("status", "400"),
                            Tag.of("status", "401"), Tag.of("status", "403"),
                            Tag.of("status", "4xx"), Tag.of("status", "5xx"))) {
                        if (meter.getId().getTags().contains(tag)) {
                            // skipping the same meter passed to this method
                            continue;
                        }
                        List<Tag> copyTags = new ArrayList<>(tags);
                        copyTags.add(tag);
                        registry.timer(meter.getId().getName(), copyTags);
                    }
                    initAllTagsInProgressFlag.set(false);
                }
            });
        };
    }

    @Bean
    public WebMvcTagsProvider getTagsProvider() {
        return new WebMvcTagsProvider() {
            @Override
            public Iterable<Tag> getTags(HttpServletRequest request, HttpServletResponse response, Object handler, Throwable exception) {
                Tag uriTag = WebMvcTags.uri(request, response);
                if ("/error".equals(uriTag.getValue())) {
                    uriTag = Tag.of("uri", String.valueOf(request.getAttribute("javax.servlet.error.request_uri")));
                }
                Tag methodTag = WebMvcTags.method(request);
                Tag exceptionTag = WebMvcTags.exception(exception);
                Tag statusTag = tagFromStatus(response);
                return Tags.of(methodTag, uriTag, exceptionTag, statusTag);
            }

            @Override
            public Iterable<Tag> getLongRequestTags(HttpServletRequest request, Object handler) {
                return Tags.of(WebMvcTags.method(request), WebMvcTags.uri(request, null));
            }
        };
    }
}
