package ru.yandex.travel.externalapi.configuration;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.externalapi.service.providers.OfferCacheInterfaceProvider;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.hotels.proto.OfferCacheServiceV1Grpc;
import ru.yandex.travel.hotels.proto.TPingRpcReq;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;

@Slf4j
@RequiredArgsConstructor
@Configuration
@EnableConfigurationProperties(OfferCacheConfigurationProperties.class)
public class OfferCacheConfiguration {
    private final OfferCacheConfigurationProperties properties;

    @Bean("OfferCacheSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean
    public HAGrpcChannelFactory offerCacheChannelFactory(
            @Qualifier("OfferCacheSupplier") ChannelSupplier channelSupplier) {

        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder
                .withPingProducer(
                        channel -> FutureUtils.buildCompletableFuture(OfferCacheServiceV1Grpc
                                .newFutureStub(channel)
                                .ping(TPingRpcReq.newBuilder().build())
                        ).thenApply(rsp -> rsp.getIsReady() ? ChannelState.READY_MASTER : ChannelState.NOT_READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(this::createChannel)
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(OfferCacheServiceV1Grpc.getPingMethod().getFullMethodName())
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(Arrays.asList(
                loggingClientInterceptor,
                defaultTimeoutClientInterceptor));
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }

    @Bean
    public OfferCacheInterfaceProvider offerCacheInterfaceProvider(
            @Qualifier("offerCacheChannelFactory") HAGrpcChannelFactory channelFactory) {
        return new OfferCacheInterfaceProvider(channelFactory);
    }

}
