package ru.yandex.travel.externalapi.endpoint;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;

import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.externalapi.endpoint.model.HotelInventoryChangesReq;
import ru.yandex.travel.externalapi.exceptions.ExternalApiBadRequestException;
import ru.yandex.travel.hotels.busbroker.model.DateRange;

public class ControllerHelpers {
    private final static int INVALIDATE_REQUEST_NIGHTS_MAX = 30;
    private final static int INVALIDATE_REQUEST_RANGES_MAX = 100;

    @Deprecated
    public static <Rsp> DeferredResult<Rsp> replyWithFuture(CompletableFuture<Rsp> f) {
        DeferredResult<Rsp> result = new DeferredResult<>();
        f.whenComplete((r, t) -> {
            if (t == null) {
                result.setResult(r);
            } else {
                Throwable realException = t;
                // Dropping extra wrappers
                while (realException instanceof CompletionException) {
                    realException = Objects.requireNonNull(realException.getCause());
                }
                result.setErrorResult(realException);
            }
        });
        return result;
    }

    public static List<DateRange> getInvalidateRequestDates(HotelInventoryChangesReq request) {
        List<DateRange> dateRanges = new ArrayList<>();
        for (var hotelUpdate : request.getCheckinDates().entrySet()) {
            int nights = hotelUpdate.getKey();
            if (nights < 1 || nights > INVALIDATE_REQUEST_NIGHTS_MAX) {
                throw new ExternalApiBadRequestException(String.format(
                        "Invalid value for Nights (valid values 1 - %d): %d", INVALIDATE_REQUEST_NIGHTS_MAX, nights
                ));
            }
            int rangesCount = hotelUpdate.getValue().size();
            if (rangesCount == 0 || rangesCount > INVALIDATE_REQUEST_RANGES_MAX) {
                throw new ExternalApiBadRequestException(String.format(
                        "Invalid array size for Nights = %d (valid size 1 - %d): %d", nights,
                        INVALIDATE_REQUEST_RANGES_MAX, rangesCount
                ));
            }
            for (var dateRange : hotelUpdate.getValue()) {
                LocalDate dateFrom = parseDateIso(dateRange.getFirstDate());
                LocalDate dateTo = parseDateIso(dateRange.getLastDate());
                if (dateFrom.isAfter(dateTo)) {
                    throw new ExternalApiBadRequestException(String.format(
                            "Invalid date range for Nights = %d. LastDate = %s should not be less than FirstDate = %s",
                            nights, dateTo, dateFrom
                    ));
                }
                for (LocalDate checkinDate : dateFrom.datesUntil(dateTo.plusDays(1)).collect(Collectors.toList())) {
                    dateRanges.add(new DateRange(checkinDate, checkinDate.plusDays(nights)));
                }
            }
        }
        return dateRanges;
    }

    public static LocalDate parseDateIso(String date) {
        try {
            return LocalDate.parse(date);
        } catch (Exception e) {
            throw new ExternalApiBadRequestException(e.getMessage());
        }
    }
}
