package ru.yandex.travel.externalapi.endpoint.google_hotels;

import java.util.concurrent.CompletableFuture;

import javax.validation.Valid;

import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.http.MediaType;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.travel.externalapi.endpoint.google_hotels.model.HintReq;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.HintRsp;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.QueryReq;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.QueryRsp;
import ru.yandex.travel.externalapi.exceptions.ExternalApiBadRequestException;
import ru.yandex.travel.externalapi.service.GoogleHotelsService;

@RestController
@RequiredArgsConstructor
@RequestMapping(value = "/google_hotels")
public class GoogleHotelsController {
    private final GoogleHotelsService googleHotelsService;

    @Bean
    public MappingJackson2XmlHttpMessageConverter mappingJackson2XmlHttpMessageConverter(Jackson2ObjectMapperBuilder builder) {
        XmlMapper xmlMapper = builder.createXmlMapper(true).build();
        xmlMapper.configure(ToXmlGenerator.Feature.WRITE_XML_DECLARATION, true);
        return new MappingJackson2XmlHttpMessageConverter(xmlMapper);
    }

    @RequestMapping(
            value = "/hint",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_XML_VALUE,
            produces = MediaType.APPLICATION_XML_VALUE
    )
    public CompletableFuture<HintRsp> hint(@Valid @RequestBody HintReq req) {
        return googleHotelsService.getHint(req);
    }

    private void validateQueryReq(QueryReq req) {
        if (req.getLatencySensitive()) {
            if (req.getContext() == null) {
                throw new ExternalApiBadRequestException("Live query must have context");
            }
            if (req.getContext().getOccupancyDetails() != null) {
                if (req.getContext().getOccupancyDetails().getNumAdults() > 10) {
                    throw new ExternalApiBadRequestException("NumAdults can not be greater then 10");
                }
                if (req.getContext().getOccupancyDetails().getChildren() != null &&
                        req.getContext().getOccupancyDetails().getChildren().stream().anyMatch(age -> age > 17)) {
                    throw new ExternalApiBadRequestException("Child's age can not be greater then 17");
                }
            }
        }
    }

    @RequestMapping(
            value = "/query",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_XML_VALUE,
            produces = MediaType.APPLICATION_XML_VALUE
    )
    public CompletableFuture<QueryRsp> query(@Valid @RequestBody QueryReq req) {
        validateQueryReq(req);
        return googleHotelsService.query(req);
    }
}
