package ru.yandex.travel.externalapi.indexes;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopScoreDocCollector;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.affiliate.TAffiliateOrder;
import ru.yandex.travel.externalapi.configuration.AffiliateOrderIndexProperties;
import ru.yandex.travel.externalapi.endpoint.affiliate.EAffiliateId;
import ru.yandex.travel.yt_lucene_index.ProtobufYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(AffiliateOrderIndexProperties.class)
@Slf4j
public class AffiliateOrderIndex {

    private static final String FIELD_AFFILIATE_ID = "iai";
    public static final String FIELD_CREATED_AT = "ica";
    public static final String FIELD_UPDATED_AT = "iua";

    public static final String FIELD_PROTO_STORE = "ps";

    private final YtLuceneIndex index;

    public AffiliateOrderIndex(AffiliateOrderIndexProperties properties) {
        if (!properties.isEnabled()) {
            log.warn("Affiliate order index disabled");
            index = null;
            return;
        }
        index = new ProtobufYtLuceneIndex<>(properties, "AffiliateIndex", TAffiliateOrder.class, this::documentProducer);
        index.start();
    }

    private Iterable<Document> documentProducer(TAffiliateOrder row) {
        Document document = new Document();
        document.add(new StringField(FIELD_AFFILIATE_ID, row.getAffiliateId(), Field.Store.YES));
        document.add(new LongPoint(FIELD_CREATED_AT, row.getCreatedAt()));
        document.add(new LongPoint(FIELD_UPDATED_AT, row.getUpdatedAt()));
        document.add(new StoredField(FIELD_PROTO_STORE, row.toByteArray()));
        return Collections.singletonList(document);
    }

    public List<TAffiliateOrder> getOrdersByDateRange(
            EAffiliateId affiliateId,
            String tsField,
            long tsFrom,
            long tsTo,
            int offset,
            int limit
    ) {
        if (index == null) {
            return List.of();
        }
        TopScoreDocCollector collector = TopScoreDocCollector.create(offset + limit);
        return index.search(searcher -> {
            Query affiliateIdQuery = new TermQuery(new Term(FIELD_AFFILIATE_ID, affiliateId.getValue()));
            Query createdAtQuery = LongPoint.newRangeQuery(tsField, tsFrom, tsTo);
            Query query = new BooleanQuery.Builder()
                    .add(new BooleanClause(affiliateIdQuery, BooleanClause.Occur.MUST))
                    .add(new BooleanClause(createdAtQuery, BooleanClause.Occur.MUST))
                    .build();
            searcher.search(query, collector);
            TopDocs topDocs = collector.topDocs(offset, limit);

            List<TAffiliateOrder> orders = new ArrayList<>();
            for (ScoreDoc scoreDoc : topDocs.scoreDocs) {
                orders.add(mapAffiliateOrderInfo(searcher.doc(scoreDoc.doc)));
            }
            return orders;
        });
    }

    private TAffiliateOrder mapAffiliateOrderInfo(Document doc) {
        try {
            return TAffiliateOrder.parser().parseFrom(doc.getField(FIELD_PROTO_STORE).binaryValue().bytes);
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException(e);
        }
    }
}
