package ru.yandex.travel.externalapi.indexes;

import java.nio.file.Path;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StringField;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.externalapi.configuration.GoogleHotelsBlackListIndexProperties;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

// TODO(mikhalych): Перевести на ProtobufYtLuceneIndex
@Component
@EnableConfigurationProperties(GoogleHotelsBlackListIndexProperties.class)
@Slf4j
public class GoogleHotelsBlackListIndex {
    private static final int INDEX_STRUCTURE_VERSION = 0;

    private static final String FIELD_SLUG = "Slug";

    private final YtLuceneIndex index;

    public GoogleHotelsBlackListIndex(GoogleHotelsBlackListIndexProperties properties) {
        if (properties.isEnabled()) {
            index = new YsonYtLuceneIndex(properties, "SlugToPermalinkIndex", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                String slug = row.getString(FIELD_SLUG);
                document.add(new StringField(FIELD_SLUG, slug, Field.Store.YES));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Slug to permalink index disabled");
            index = null;
        }
        log.info(Path.of(properties.getIndexPath()).toAbsolutePath().normalize().toString());
    }

    @PostConstruct
    public void init() {
        if (index != null) {
            index.start();
        }
    }

    @PreDestroy
    public void destroy() {
        if (index != null) {
            index.stop();
        }
    }

    private boolean isReady() {
        return index != null && index.isReady();
    }

    public Set<String> getAllSlugs() {
        if (!isReady()) {
            return Set.of();
        }

        Set<String> slugs = new HashSet<>();
        index.forEachDocument(doc -> slugs.add(doc.getField(FIELD_SLUG).stringValue()));
        return slugs;
    }
}
