package ru.yandex.travel.externalapi.indexes;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.externalapi.configuration.SlugToPermalinkIndexProperties;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;


// TODO(mikhalych): перевести на MapPersistentConfig
@Component
@EnableConfigurationProperties(SlugToPermalinkIndexProperties.class)
@Slf4j
public class SlugToPermalinkIndex {
    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_SLUG = "main_slug";
    private static final String FIELD_PERMALINK = "cluster_permalink";

    private final YtLuceneIndex index;

    public SlugToPermalinkIndex(SlugToPermalinkIndexProperties properties) {
        if (properties.isEnabled()) {
            index = new YsonYtLuceneIndex(properties, "SlugToPermalinkIndex", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                String slug = row.getString(FIELD_SLUG);
                long permalink = row.getLong(FIELD_PERMALINK);

                document.add(new StringField(FIELD_SLUG, slug, Field.Store.YES));
                document.add(new LongPoint(FIELD_PERMALINK, permalink));
                document.add(new StoredField(FIELD_PERMALINK, permalink));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Slug to permalink index disabled");
            index = null;
        }
        log.info(Path.of(properties.getIndexPath()).toAbsolutePath().normalize().toString());
    }

    @PostConstruct
    public void init() {
        if (index != null) {
            index.start();
        }
    }

    @PreDestroy
    public void destroy() {
        if (index != null) {
            index.stop();
        }
    }

    private boolean isReady() {
        return index != null && index.isReady();
    }

    public List<String> getAllSlugs() {
        if (!isReady()) {
            return List.of();
        }
        List<String> slugs = new ArrayList<>();
        index.forEachDocument(doc -> slugs.add(doc.getField(FIELD_SLUG).stringValue()));
        return slugs;
    }

    public Long getPermalinkBySlug(String slug) {
        if (!isReady()) {
            return null;
        }

        return index.search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_SLUG, slug));
            TopDocs topDocs = searcher.search(query, 1);

            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                return document.getField(FIELD_PERMALINK).numericValue().longValue();
            }
            return null;
        });
    }

    public String getSlugByPermalink(Long permalink) {
        if (!isReady()) {
            return null;
        }

        return index.search(searcher -> {
            Query query = LongPoint.newExactQuery(FIELD_PERMALINK, permalink);
            TopDocs topDocs = searcher.search(query, 1);

            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                return document.getField(FIELD_SLUG).stringValue();
            }
            return null;
        });
    }
}
