package ru.yandex.travel.externalapi.security;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.servlet.HandlerInterceptor;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.inside.passport.YandexAccounts;
import ru.yandex.inside.passport.blackbox.Blackbox;
import ru.yandex.inside.passport.blackbox.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox.protocol.BlackboxFatalException;
import ru.yandex.inside.passport.blackbox.protocol.BlackboxOAuthStatus;
import ru.yandex.inside.passport.blackbox.protocol.BlackboxResponse;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
public class OAuthAuthorizationInterceptor implements HandlerInterceptor {

    public static final String AUTHORIZATION_HEADER_NAME = "Authorization";
    private static final Pattern AUTHORIZATION_HEADER_PATTERN = Pattern.compile("Bearer ([^ ]+)");

    private final TvmWrapper tvm;
    private final Blackbox blackbox;
    private final AuthorizationProperties properties;

    public OAuthAuthorizationInterceptor(TvmWrapper tvm, Blackbox blackbox, AuthorizationProperties properties) {
        this.tvm = tvm;
        this.blackbox = blackbox;
        this.properties = properties;
        Preconditions.checkNotNull(tvm);
    }


    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        try {
            String authorizationHeader = request.getHeader(AUTHORIZATION_HEADER_NAME);
            if (authorizationHeader != null) {
                Matcher matcher = AUTHORIZATION_HEADER_PATTERN.matcher(authorizationHeader);
                if (matcher.find()) {
                    String token = matcher.group(1);
                    try {
                        BlackboxResponse blackboxResponse = blackbox.oAuth(
                                IpAddress.parse(request.getRemoteAddr()),
                                token,
                                List.of(),
                                Tuple2List.fromPairs(YandexAccounts.SERVICE_TICKET, tvm.getServiceTicket(properties.getTvmBlackboxAlias())));
                        if (blackboxResponse.getStatus() == BlackboxOAuthStatus.VALID.getId()
                                && blackboxResponse.getOAuthInfo().isPresent()
                                && blackboxResponse.getOAuthInfo().get().getScopes().containsTs(properties.getPartnerScope())) {
                            for (AuthorizationProperties.PartnerData partnerData : properties.getPartnerData().values()) {
                                if (request.getRequestURI().startsWith(partnerData.getUrlPrefix()) &&
                                        blackboxResponse.getOAuthInfo().get().getClientId().equals(partnerData.getAllowedClientId())) {
                                    return true;
                                }
                            }
                        }
                    } catch (BlackboxException e) {
                        if (e instanceof BlackboxFatalException) {
                            //Rethrow fatal Exception
                            throw e;
                        } else {
                            //log OAuth exception: expired token, invalid token, etc.
                            log.info("Non fatal BlackboxException occurred: Class: {}; Message: {}", e.getClass(), e.getMessage());
                        }
                    }
                }
            }
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Authorization token is invalid or missing");
            return false;
        } catch (Exception e) {
            //Log exception and return HTTP 500
            log.error("Exception while request authorization occurred: ", e);
            throw e;
        }
    }
}
