package ru.yandex.travel.externalapi.service;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.List;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.affiliate.TAffiliateOrder;
import ru.yandex.travel.externalapi.configuration.ExternalApiProperties;
import ru.yandex.travel.externalapi.endpoint.ControllerHelpers;
import ru.yandex.travel.externalapi.endpoint.affiliate.EAffiliateId;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.AffiliateOrder;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.AffiliateOrdersRsp;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.PageInfo;
import ru.yandex.travel.externalapi.exceptions.ExternalApiBadRequestException;
import ru.yandex.travel.externalapi.indexes.AffiliateOrderIndex;

@Service
@AllArgsConstructor
public class AffiliateService {
    private final static int LIMIT_MAX = 10_000;

    private final ExternalApiProperties externalApiProperties;
    private final AffiliateOrderIndex orderIndex;
    private final OrderMapper orderMapper;

    public AffiliateOrdersRsp getOrdersData(
            EAffiliateId affiliateId,
            String dateFrom,
            String dateTo,
            int offset,
            int limit
    ) {
        return getOrdersDataCommon(affiliateId, dateFrom, dateTo, offset, limit, AffiliateOrderIndex.FIELD_CREATED_AT);
    }

    public AffiliateOrdersRsp getUpdatedOrdersData(
            EAffiliateId affiliateId,
            String dateFrom,
            String dateTo,
            int offset,
            int limit
    ) {
        return getOrdersDataCommon(affiliateId, dateFrom, dateTo, offset, limit, AffiliateOrderIndex.FIELD_UPDATED_AT);
    }

    private AffiliateOrdersRsp getOrdersDataCommon(
            EAffiliateId affiliateId,
            String reqDateFrom,
            String reqDateTo,
            int offset,
            int limit,
            String tsField
    ) {
        if (limit > LIMIT_MAX) {
            throw new ExternalApiBadRequestException(String.format(
                    "Invalid value for limit (valid values 1 - %d): %d", LIMIT_MAX, limit
            ));
        }
        LocalDate dateFrom = ControllerHelpers.parseDateIso(reqDateFrom);
        LocalDate dateTo = ControllerHelpers.parseDateIso(reqDateTo);
        if (!dateTo.isAfter(dateFrom)) {
            throw new ExternalApiBadRequestException(String.format(
                    "Invalid date range. dateTo = %s should be after dateFrom = %s", dateTo, dateFrom
            ));
        }
        long tsFrom = dateFrom.toEpochSecond(LocalTime.MIDNIGHT, externalApiProperties.getDefaultZoneOffset());
        long tsTo = dateTo.plusDays(1).toEpochSecond(LocalTime.MIDNIGHT, externalApiProperties.getDefaultZoneOffset());
        List<TAffiliateOrder> indexOrders = orderIndex.getOrdersByDateRange(
                affiliateId, tsField, tsFrom, tsTo, offset, limit
        );
        List<AffiliateOrder> orders = indexOrders.stream().map(orderMapper::mapOrder).collect(Collectors.toList());

        offset += orders.size();
        return AffiliateOrdersRsp.builder()
                .orders(orders)
                .page(PageInfo.builder()
                        .offset(offset)
                        .limit(limit)
                        .build()
                )
                .build();
    }
}
