package ru.yandex.travel.externalapi.service;

import java.time.LocalDate;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.externalapi.configuration.ExternalApiProperties;
import ru.yandex.travel.externalapi.endpoint.bnovo.model.HotelStatus;
import ru.yandex.travel.externalapi.endpoint.bnovo.model.HotelStatusRsp;
import ru.yandex.travel.externalapi.endpoint.bnovo.model.UnpublishedReason;
import ru.yandex.travel.externalapi.endpoint.model.HotelDetailsChangedReq;
import ru.yandex.travel.externalapi.service.providers.AdministratorInterfaceProvider;
import ru.yandex.travel.hotels.administrator.grpc.proto.EHotelStatus;
import ru.yandex.travel.hotels.administrator.grpc.proto.EUnpublishedReason;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedRsp;
import ru.yandex.travel.hotels.busbroker.BusBrokerClient;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Service
public class BnovoHotelsService extends PartnerHotelsService {

    private final ExternalApiProperties externalApiProperties;

    public BnovoHotelsService(AdministratorInterfaceProvider administratorInterfaceProvider,
                              ExternalApiProperties externalApiProperties, BusBrokerClient busBrokerClient) {
        super(administratorInterfaceProvider, busBrokerClient);
        this.externalApiProperties = externalApiProperties;
    }

    public CompletableFuture<THotelDetailsChangedRsp> notifyHotelDetailsChanged(HotelDetailsChangedReq request) {
        return super.notifyHotelDetailsChanged(request, EPartnerId.PI_BNOVO);
    }

    public CompletableFuture<HotelStatusRsp> getHotelStatus(String hotelCode) {
        return super.getHotelStatus(hotelCode, EPartnerId.PI_BNOVO).thenApply(hotelStatusResponse -> {
            if (hotelStatusResponse.getHotelStatus().equals(EHotelStatus.H_NOT_FOUND)) {
                return null;
            }

            HotelStatusRsp response = new HotelStatusRsp();
            HotelStatus hotelStatus = new HotelStatus();
            hotelStatus.setHotelCode(hotelStatusResponse.getHotelCode());
            hotelStatus.setUnpublishedReason(mapUnpublishedReason(hotelStatusResponse.getUnpublishedReason()));
            hotelStatus.setStatus(mapHotelStatus(hotelStatusResponse.getHotelStatus()));
            if (StringUtils.isNotBlank(hotelStatusResponse.getAgreementId())) {
                hotelStatus.setAgreementId(hotelStatusResponse.getAgreementId());
            }
            if (hotelStatusResponse.hasAgreementFrom()) {
                hotelStatus.setAgreementFrom(LocalDate.ofInstant(ProtoUtils.toInstant(hotelStatusResponse.getAgreementFrom()), externalApiProperties.getDefaultZoneOffset()).toString());
            }
            if (hotelStatusResponse.getPermalink() != 0) {
                String url = String.format(externalApiProperties.getPortalHotelUrlTemplate(), hotelStatusResponse.getPermalink());
                hotelStatus.setHotelUrl(url);
            }
            response.setHotelStatus(hotelStatus);
            return response;
        });
    }

    private UnpublishedReason mapUnpublishedReason(EUnpublishedReason unpublishedReason) {
        switch (unpublishedReason) {
            case UR_NONE:
                return UnpublishedReason.NONE;
            case UR_NO_AGREEMENT:
                return UnpublishedReason.NO_AGREEMENT;
            case UR_SUSPENDED:
                return UnpublishedReason.SUSPENDED;
            case UR_UPDATING:
                return UnpublishedReason.UPDATING;
            case UR_DELAYED:
                return UnpublishedReason.DELAYED;
            default:
                throw new RuntimeException("Unexpected unpublished reason:" + unpublishedReason);
        }
    }
}
