package ru.yandex.travel.externalapi.service;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.externalapi.configuration.GoogleHotelsServiceProperties;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.HintReq;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.HintRsp;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.QueryReq;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.QueryRsp;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.hint.ExactItinerariesHintItem;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.hint.HintItem;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.PackageData;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.PhotoUrl;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.PropertyDataSet;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.Rate;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.Refundable;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.Result;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.RoomBundle;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.RoomData;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.RspOccupancyDetails;
import ru.yandex.travel.externalapi.endpoint.google_hotels.model.query.Text;
import ru.yandex.travel.externalapi.exceptions.ExternalApiBadRequestException;
import ru.yandex.travel.externalapi.indexes.GoogleHotelsBlackListIndex;
import ru.yandex.travel.externalapi.indexes.SlugToPermalinkIndex;
import ru.yandex.travel.externalapi.service.providers.OfferCacheInterfaceProvider;
import ru.yandex.travel.hotels.common.Ages;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.TPrice;
import ru.yandex.travel.hotels.offercache.api.TReadReq;
import ru.yandex.travel.hotels.offercache.api.TReadResp;
import ru.yandex.travel.hotels.offercache.api.TRefundRule;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERefundType;

@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(GoogleHotelsServiceProperties.class)
@Service
public class GoogleHotelsService {
    private static final Integer LENGTH_OF_STAY = 2;
    private static final String DEBUG_ID = "ext-api_google-hotels";
    private static final Integer DEFAULT_NUM_ADULTS = 2;
    private static final ZoneId UTC = ZoneId.of("UTC");
    private static final String GEO_CLIENT_ID = "external-api";
    private static final String GEO_ORIGIN = "travel-ext-api-google-hotels";

    private static final DateTimeFormatter GOOGLE_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final DateTimeFormatter GOOGLE_TIME_FORMAT = DateTimeFormatter.ofPattern("hh:mm:ss");

    private final OfferCacheInterfaceProvider offerCacheInterfaceProvider;
    private final SlugToPermalinkIndex slugToPermalinkIndex;
    private final GoogleHotelsBlackListIndex blackListIndex;
    private final GoogleHotelsServiceProperties serviceProperties;

    public CompletableFuture<HintRsp> getHint(HintReq req) {
        String checkInDate = LocalDateTime.ofInstant(req.getLastFetchTime(), UTC).format(GOOGLE_DATE_FORMAT);

        Set<String> blackListedSlugs = blackListIndex.getAllSlugs();

        List<HintItem> hintItems = slugToPermalinkIndex.getAllSlugs().stream()
                .filter(slug -> !blackListedSlugs.contains(slug))
                .map(slug -> ExactItinerariesHintItem.builder()
                        .property(slug)
                        .stay(ExactItinerariesHintItem.HintStay.builder()
                                .checkInDate(checkInDate)
                                .lengthOfStay(LENGTH_OF_STAY)
                                .build())
                        .build())
                .collect(Collectors.toList());

        return CompletableFuture.supplyAsync(() -> HintRsp.builder().items(hintItems).build());
    }

    private Refundable mapRefundable(TPrice price, Instant now) {
        if (price.getRefundRuleList().isEmpty()) {
            return null;
        }
        TRefundRule refundRule = price.getRefundRuleList().stream()
                .filter(x -> (
                        x.getType() == ERefundType.RT_FULLY_REFUNDABLE ||
                                x.getType() == ERefundType.RT_REFUNDABLE_WITH_PENALTY
                ))
                .max(Comparator.comparing(TRefundRule::getEndsAtTimestampSec))
                .orElse(price.getRefundRule(0));
        switch (refundRule.getType()) {
            case RT_FULLY_REFUNDABLE:
            case RT_REFUNDABLE_WITH_PENALTY:
                Refundable.RefundableBuilder refundableBuilder = Refundable.builder();
                if (refundRule.hasEndsAtTimestampSec()) {
                    Instant refundEndsTime = Instant.ofEpochSecond(refundRule.getEndsAtTimestampSec());
                    refundableBuilder.refundable_until_days(Duration.between(now, refundEndsTime).toDaysPart());
                    refundableBuilder.refundable_until_time(LocalDateTime.ofInstant(refundEndsTime, UTC).format(GOOGLE_TIME_FORMAT));
                }
                return refundableBuilder.build();
            default:
                return Refundable.builder().available(false).build();
        }
    }

    private String makeRoomId(String id) {
        return String.format("room_%s", id);
    }

    private String makePackageId(String id) {
        return String.format("package_%s", id);
    }

    private String makeRatePlanId(String roomId, String packageId) {
        return String.format("%s-%s", roomId, packageId);
    }

    private QueryRsp mapQueryRsp(TReadResp rsp, Instant now, Ages occupancyAges) {
        List<PropertyDataSet> properties = new ArrayList<>();
        List<Result> results = new ArrayList<>();
        Rate emptyRate = Rate.builder()
                .currency(rsp.getCurrency().toString())
                .value("00.00")
                .build();
        Integer occupancy = occupancyAges.getAdults() + occupancyAges.getChildrenAges().size();
        RspOccupancyDetails occupancyDetails = RspOccupancyDetails.builder()
                .numAdults(occupancyAges.getAdults())
                .children(occupancyAges.getChildrenAges().stream()
                        .map(childAge -> RspOccupancyDetails.Child.builder()
                                .age(childAge)
                                .build())
                        .collect(Collectors.toList())
                )
                .build();

        rsp.getHotelsMap().forEach((permalink, hotel) -> {
            String slug = slugToPermalinkIndex.getSlugByPermalink(permalink);

            PropertyDataSet property = PropertyDataSet.builder()
                    .property(slug)
                    .roomData(hotel.getPermaroomsList().stream()
                            .map(permaroom -> {
                                RoomData.RoomDataBuilder roomDataBuilder = RoomData.builder()
                                        .roomID(makeRoomId(permaroom.getId()))
                                        .names(List.of(Text.builder().text(permaroom.getName()).build()));
                                if (serviceProperties.isEnablePhotos()) {
                                    roomDataBuilder.photoUrls(permaroom.getPhotosList().stream()
                                            .map(photo -> PhotoUrl.builder()
                                                    .url(String.format(photo.getUrlTemplate(), "orig"))
                                                    .build())
                                            .distinct()
                                            .collect(Collectors.toList()));
                                }
                                return roomDataBuilder.build();
                            })
                            .collect(Collectors.toList()))
                    .packageData(hotel.getPermaroomsList().stream()
                            .map(permaroom -> PackageData.builder()
                                    .names(List.of(Text.builder().text(permaroom.getName()).build()))
                                    .packageID(makePackageId(permaroom.getId()))
                                    .internetIncluded(permaroom.getEnumFeaturesList().stream()
                                            .anyMatch(x -> x.getId().equals("wifi") && x.getValueId().equals("free"))
                                    )
                                    .build())
                            .collect(Collectors.toList()))
                    .build();
            List<RoomBundle> roomBundles = hotel.getPricesList().stream()
                    .collect(Collectors.groupingBy(
                            TPrice::getPermaroomId,
                            Collectors.minBy(Comparator.comparingInt(TPrice::getPrice)))
                    )
                    .values().stream()
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .map(price -> {
                        String roomId = makeRoomId(price.getPermaroomId());
                        String packageId = makePackageId(price.getPermaroomId());
                        String ratePlanId = makeRatePlanId(roomId, packageId);

                        return RoomBundle.builder()
                                .baseRate(Rate.builder()
                                        .currency(rsp.getCurrency().toString())
                                        .value(String.format("%d.00", price.getPrice()))
                                        .build())
                                .tax(emptyRate)
                                .otherFees(emptyRate)
                                .roomID(roomId)
                                .packageID(packageId)
                                .ratePlanID(ratePlanId)
                                .occupancy(occupancy)
                                .occupancyDetails(occupancyDetails)
                                .refundable(mapRefundable(price, now))
                                .breakfastIncluded(price.getBreakfastIncluded())
                                .build();
                    })
                    .collect(Collectors.toList());

            if (!roomBundles.isEmpty()) {
                results.add(Result.builder()
                        .checkin(rsp.getDate())
                        .nights(rsp.getNights())
                        .property(slug)
                        .roomBundles(roomBundles)
                        .build());
            }

            properties.add(property);
        });

        return QueryRsp.builder()
                .id(UUID.randomUUID().toString())
                .timestamp(now.toString())
                .properties(properties)
                .results(results)
                .build();
    }

    public CompletableFuture<QueryRsp> query(QueryReq req) {
        Instant now = Instant.now();
        LocalDate today = LocalDateTime.ofInstant(now, UTC).toLocalDate();

        LocalDate checkInDate = LocalDate.parse(req.getCheckIn());

        if (checkInDate.isBefore(today)) {
            throw new ExternalApiBadRequestException("Wrong Checkin date");
        }
        LocalDate checkOutDate = checkInDate.plusDays(req.getNights());

        Integer numAdults = DEFAULT_NUM_ADULTS;
        List<Integer> childrenAges = List.of();
        boolean useSearcher = false;

        if (req.getLatencySensitive()) {
            if (req.getContext().getOccupancyDetails() == null) {
                if (req.getContext().getOccupancy() != null) {
                    numAdults = req.getContext().getOccupancy();
                }
            } else {
                if (req.getContext().getOccupancyDetails().getNumAdults() != null) {
                    numAdults = req.getContext().getOccupancyDetails().getNumAdults();
                }
                if (req.getContext().getOccupancyDetails().getChildren() != null &&
                        !req.getContext().getOccupancyDetails().getChildren().isEmpty()) {
                    childrenAges = req.getContext().getOccupancyDetails().getChildren();
                }
                useSearcher = true;
            }
        }
        Ages occupancyAges = Ages.build(numAdults, childrenAges);

        List<String> hotelIDs = req.getProperties().stream()
                .map(slugToPermalinkIndex::getPermalinkBySlug)
                .filter(Objects::nonNull)
                .map(Object::toString)
                .collect(Collectors.toList());
        if (hotelIDs.isEmpty()) {
            return CompletableFuture.supplyAsync(() -> QueryRsp.builder().build());
        }

        TReadReq.Builder ocReqBuilder = OfferCacheRequestParams.prepareBuilder(
                        checkInDate, checkOutDate, occupancyAges.getAdults(), occupancyAges.getChildrenAges(),
                        useSearcher, DEBUG_ID, true, false
                )
                .addAllSHotelId(hotelIDs)
                .setFull(true)
                .setEnableCatRoom(true)
                .setAutoCatRoomOnlyForBoY(true)
                .setUseNewCatRoom(true)
                .setShowPermaroomsWithNoOffers(true)
                .setBoYPartner(EPartnerId.PI_TRAVELLINE_VALUE)
                .setAttribution(TReadReq.TAttribution.newBuilder()
                        .setGeoClientId(GEO_CLIENT_ID)
                        .setGeoOrigin(GEO_ORIGIN)
                        .build());

        TReadReq ocRequest = OfferCacheRequestParams.build(ocReqBuilder).getRequest();
        return FutureUtils.buildCompletableFuture(offerCacheInterfaceProvider.provideInterface().read(ocRequest))
                .thenApply(x -> mapQueryRsp(x, now, occupancyAges));
    }
}
