package ru.yandex.travel.externalapi.service;

import java.time.Instant;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.affiliate.TAffiliateOrder;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.AffiliateOrder;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.AviaAffiliateOrder;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.EAffiliateOrderType;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.EOrderState;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.HotelAffiliateOrder;
import ru.yandex.travel.externalapi.endpoint.affiliate.model.TrainAffiliateOrder;

@Component
@Slf4j
public class OrderMapper {
    private AffiliateOrder mapCommonOrder(AffiliateOrder.AffiliateOrderBuilder affiliateOrderBuilder,
                                          TAffiliateOrder orderInfo) {
        return affiliateOrderBuilder
                .partnerUid(orderInfo.getAffiliateUid())
                .partnerOrderId(orderInfo.getPartnerOrderId())
                .state(EOrderState.valueOf(orderInfo.getState()))
                .createdAt(Instant.ofEpochSecond(orderInfo.getCreatedAt()))
                .updatedAt(Instant.ofEpochSecond(orderInfo.getUpdatedAt()))
                .orderAmount(orderInfo.getOrderAmount())
                .feeAmount(orderInfo.getFeeAmount())
                .feeAmountWithoutNds(orderInfo.getFeeAmountWithoutNds())
                .bookedOnYandex(orderInfo.getBookedOnYandex())
                .currency(orderInfo.getCurrency())
                .type(orderInfo.getCategory())
                .build();
    }

    private AffiliateOrder mapHotelOrder(TAffiliateOrder orderInfo) {
        return mapCommonOrder(
                HotelAffiliateOrder.builder()
                        .checkinDate(orderInfo.getCheckIn())
                        .checkoutDate(orderInfo.getCheckOut())
                        .hotelName(orderInfo.getHotelName())
                        .country(orderInfo.getHotelCountry())
                        .city(orderInfo.getHotelCity()),
                orderInfo
        );
    }

    private AffiliateOrder mapTrainOrder(TAffiliateOrder orderInfo) {
        return mapCommonOrder(
                TrainAffiliateOrder.builder()
                        .stationDeparture(orderInfo.getStationDeparture())
                        .stationArrival(orderInfo.getStationArrival())
                        .cityDeparture(orderInfo.getCityDeparture())
                        .cityArrival(orderInfo.getCityArrival())
                        .tripType(orderInfo.getTripType())
                        .carrierName(orderInfo.getCarrierName())
                        .travelDate(Instant.ofEpochSecond(orderInfo.getTravelDate())),
                orderInfo
        );
    }

    private AffiliateOrder mapAviaOrder(TAffiliateOrder orderInfo) {
        return mapCommonOrder(
                AviaAffiliateOrder.builder()
                        .stationDeparture(orderInfo.getStationDeparture())
                        .stationArrival(orderInfo.getStationArrival())
                        .cityDeparture(orderInfo.getCityDeparture())
                        .cityArrival(orderInfo.getCityArrival())
                        .tripType(orderInfo.getTripType())
                        .carrierName(orderInfo.getCarrierName())
                        .travelDate(Instant.ofEpochSecond(orderInfo.getTravelDate())),
                orderInfo
        );
    }

    public AffiliateOrder mapOrder(TAffiliateOrder orderInfo) {
        EAffiliateOrderType orderType = EAffiliateOrderType.fromString(orderInfo.getCategory());
        switch (orderType) {
            case HOTELS:
                return mapHotelOrder(orderInfo);
            case TRAIN:
                return mapTrainOrder(orderInfo);
            case AVIA:
                return mapAviaOrder(orderInfo);
            default:
                throw new RuntimeException("Unknown order type: " + orderInfo.getCategory());
        }
    }
}
