package ru.yandex.travel.externalapi.service;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.externalapi.endpoint.ControllerHelpers;
import ru.yandex.travel.externalapi.endpoint.model.AcceptAgreementReq;
import ru.yandex.travel.externalapi.endpoint.model.AcceptAgreementRsp;
import ru.yandex.travel.externalapi.endpoint.model.AcceptAgreementStatus;
import ru.yandex.travel.externalapi.endpoint.model.AgreementStatusType;
import ru.yandex.travel.externalapi.endpoint.model.CheckAgreementRsp;
import ru.yandex.travel.externalapi.endpoint.model.HotelDetailsChangedReq;
import ru.yandex.travel.externalapi.endpoint.model.HotelInventoryChangesReq;
import ru.yandex.travel.externalapi.endpoint.model.HotelStatusEnum;
import ru.yandex.travel.externalapi.exceptions.ExternalApiBadRequestException;
import ru.yandex.travel.externalapi.service.providers.AdministratorInterfaceProvider;
import ru.yandex.travel.hotels.administrator.grpc.proto.EAgreementStatusType;
import ru.yandex.travel.hotels.administrator.grpc.proto.EHotelStatus;
import ru.yandex.travel.hotels.administrator.grpc.proto.TAcceptAgreementReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.TCheckAgreementReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedRsp;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelStatusReq;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelStatusRsp;
import ru.yandex.travel.hotels.busbroker.BusBrokerClient;
import ru.yandex.travel.hotels.busbroker.model.DateRange;
import ru.yandex.travel.hotels.proto.EOfferInvalidationSource;
import ru.yandex.travel.hotels.proto.EPartnerId;

@RequiredArgsConstructor
@Slf4j
public class PartnerHotelsService {

    private final AdministratorInterfaceProvider administratorInterfaceProvider;
    private final BusBrokerClient busBrokerClient;
    private final Counter invalidationErrors = Metrics.counter("offer-invalidation.errors");

    protected CompletableFuture<THotelDetailsChangedRsp> notifyHotelDetailsChanged(HotelDetailsChangedReq request,
                                                                                   EPartnerId partnerId) {
        return FutureUtils.buildCompletableFuture(administratorInterfaceProvider.provideInterface().hotelDetailsChanged(THotelDetailsChangedReq
                .newBuilder()
                .setHotelCode(request.getHotelCode())
                .setPartnerId(partnerId)
                .build()));
    }

    protected CompletableFuture<THotelStatusRsp> getHotelStatus(String hotelCode, EPartnerId partnerId) {
        return FutureUtils.buildCompletableFuture(administratorInterfaceProvider.provideInterface()
                .hotelStatus(THotelStatusReq.newBuilder()
                        .setHotelCode(hotelCode)
                        .setPartnerId(partnerId)
                        .build()));
    }


    public CompletableFuture<AcceptAgreementRsp> acceptAgreement(AcceptAgreementReq request, EPartnerId partnerId) {
        return FutureUtils.buildCompletableFuture(administratorInterfaceProvider.provideInterface()
                .acceptAgreement(TAcceptAgreementReq.newBuilder()
                        .setPartnerId(partnerId)
                        .setInn(request.getInn())
                        .setHotelCode(request.getHotelCode())
                        .setContactName(request.getContactInfo().getName())
                        .setContactPosition(request.getContactInfo().getPosition())
                        .setContactPhone(request.getContactInfo().getPhone())
                        .setContactEmail(request.getContactInfo().getEmail())
                        .build())).thenApply(response -> {
                            switch (response.getStatus()) {
                                case AAS_SUCCESS:
                                    return AcceptAgreementRsp.builder()
                                            .status(AcceptAgreementStatus.SUCCESS)
                                            .build();
                                case AAS_FAILURE:
                                    return AcceptAgreementRsp.builder()
                                            .status(AcceptAgreementStatus.FAILURE)
                                            .message(response.getMessage())
                                            .build();
                                default:
                                    throw new RuntimeException("Unexpected accept agreement status: " + response.getStatus());
                            }
        });
    }

    public CompletableFuture<CheckAgreementRsp> checkAgreement(String inn) {
        return FutureUtils.buildCompletableFuture(administratorInterfaceProvider.provideInterface()
                .agreementStatus(TCheckAgreementReq.newBuilder()
                        .setInn(inn)
                        .build())).thenApply(response -> CheckAgreementRsp.builder()
                .agreement(mapAgreementStatusType(response.getAgreementStatus()))
                .build());
    }

    public CompletableFuture<Object> hotelInventoryChanges(
            HotelInventoryChangesReq request,
            EPartnerId partnerId,
            EOfferInvalidationSource offerInvalidationSource
    ) {
        if (request.getAccountId() == null) {
            throw new ExternalApiBadRequestException("No AccountId in request");
        }
        String originalId = request.getAccountId().toString();
        List<DateRange> dates;
        dates = ControllerHelpers.getInvalidateRequestDates(request);
        return busBrokerClient.invalidateOffersByCheckInOut(partnerId, originalId, dates, offerInvalidationSource)
                .thenApply(rsp -> {
                    if (rsp != null && rsp.getError() != null && !rsp.getError().isBlank()) {
                        throw new RuntimeException(rsp.getError());
                    }
                    return null;
                })
                .whenComplete((rsp, exception) -> {
                    if (exception != null) {
                        invalidationErrors.increment();
                        log.error("Failed to send invalidation event");
                    }
                });
    }

    protected AgreementStatusType mapAgreementStatusType(EAgreementStatusType statusType) {
        switch (statusType) {
            case AS_FOUND:
                return AgreementStatusType.FOUND;
            case AS_NOT_FOUND:
                return AgreementStatusType.NOT_FOUND;
            case AS_BLOCKED:
                return AgreementStatusType.BLOCKED;
            default:
                throw new RuntimeException("Unexpected agreement status: " + statusType);
        }
    }

    protected HotelStatusEnum mapHotelStatus(EHotelStatus hotelStatus) {
        switch (hotelStatus) {
            case H_PUBLISHING:
                return HotelStatusEnum.PUBLISHING;
            case H_PUBLISHED:
                return HotelStatusEnum.PUBLISHED;
            case H_UNPUBLISHED:
                return HotelStatusEnum.UNPUBLISHED;
            default:
                throw new RuntimeException("Unexpected hotel status: " + hotelStatus);
        }
    }
}
