package ru.yandex.travel.externalapi.service;

import java.time.LocalDate;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.externalapi.configuration.ExternalApiProperties;
import ru.yandex.travel.externalapi.endpoint.model.HotelDetailsChangedReq;
import ru.yandex.travel.externalapi.endpoint.travelline.model.BaseResponse;
import ru.yandex.travel.externalapi.endpoint.travelline.model.DisconnectionReason;
import ru.yandex.travel.externalapi.endpoint.travelline.model.HotelStatus;
import ru.yandex.travel.externalapi.endpoint.travelline.model.HotelStatusRsp;
import ru.yandex.travel.externalapi.endpoint.travelline.model.ResponseErrorCode;
import ru.yandex.travel.externalapi.service.providers.AdministratorInterfaceProvider;
import ru.yandex.travel.hotels.administrator.grpc.proto.EHotelStatus;
import ru.yandex.travel.hotels.administrator.grpc.proto.EUnpublishedReason;
import ru.yandex.travel.hotels.administrator.grpc.proto.THotelDetailsChangedRsp;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Service
public class TravellineHotelsService extends PartnerHotelsService {

    private final ExternalApiProperties externalApiProperties;

    public TravellineHotelsService(AdministratorInterfaceProvider administratorInterfaceProvider,
                                   ExternalApiProperties externalApiProperties) {
        super(administratorInterfaceProvider, null);
        this.externalApiProperties = externalApiProperties;
    }

    public CompletableFuture<THotelDetailsChangedRsp> notifyHotelDetailsChanged(HotelDetailsChangedReq request) {
        return super.notifyHotelDetailsChanged(request, EPartnerId.PI_TRAVELLINE);
    }

    public CompletableFuture<HotelStatusRsp> getHotelStatus(String hotelCode) {
        return super.getHotelStatus(hotelCode, EPartnerId.PI_TRAVELLINE).thenApply(hotelStatusResponse -> {
            HotelStatusRsp response = new HotelStatusRsp();
            if (hotelStatusResponse.getHotelStatus().equals(EHotelStatus.H_NOT_FOUND)) {
                response.setErrors(List.of(new BaseResponse.ResponseError(ResponseErrorCode.HOTEL_NOT_FOUND)));
            } else {
                HotelStatus hotelStatus = new HotelStatus();
                hotelStatus.setHotelRef(new HotelStatus.HotelRef(hotelStatusResponse.getHotelCode()));
                hotelStatus.setDisconnectionReason(mapUnpublishedReason(hotelStatusResponse.getUnpublishedReason()));
                hotelStatus.setStatus(mapHotelStatus(hotelStatusResponse.getHotelStatus()));
                if (StringUtils.isNotBlank(hotelStatusResponse.getAgreementId())) {
                    hotelStatus.setAgreementId(hotelStatusResponse.getAgreementId());
                }
                if (hotelStatusResponse.hasAgreementFrom()) {
                    hotelStatus.setAgreementFrom(LocalDate.ofInstant(ProtoUtils.toInstant(hotelStatusResponse.getAgreementFrom()), externalApiProperties.getDefaultZoneOffset()).toString());
                }
                if (hotelStatusResponse.getPermalink() != 0) {
                    String url = String.format(externalApiProperties.getPortalHotelUrlTemplate(), hotelStatusResponse.getPermalink());
                    hotelStatus.setHotelUrl(url);
                }
                response.setHotelStatus(hotelStatus);
            }
            return response;
        });
    }

    private DisconnectionReason mapUnpublishedReason(EUnpublishedReason unpublishedReason) {
        switch (unpublishedReason) {
            case UR_NONE:
                return DisconnectionReason.NONE;
            case UR_NO_AGREEMENT:
                return DisconnectionReason.NO_AGREEMENT;
            case UR_SUSPENDED:
                return DisconnectionReason.SUSPENDED;
            case UR_UPDATING:
                return DisconnectionReason.UPDATING;
            case UR_DELAYED:
                return DisconnectionReason.DELAYED;
            default:
                throw new RuntimeException("Unexpected rejection reason:" + unpublishedReason);
        }
    }
}
