import fglob from 'fast-glob';

import {TSource, IConfig, IUserConfig} from './IConfig';
import {configPath, onlyShowFiles} from '../Argv/argv';
import {TFilesWithTests} from '../Runner/IRunner';

export function config(): IConfig {
    const userConfigPath = getUserConfigPath(configPath);
    const readedConfig = readConfig(userConfigPath);

    return getConfig(readedConfig);
}

export function getUserConfigPath(userConfigPath: string): string {
    try {
        return require.resolve(userConfigPath, {
            paths: [process.cwd()],
        });
    } catch (e) {
        throw new Error(`Seems like the path ${userConfigPath} is incorrect`);
    }
}

export function readConfig(_configPath: string): IUserConfig {
    return require(_configPath);
}

export function getConfig(userConfig: IUserConfig): IConfig {
    const filesWithTests = findFilesWithTests(
        userConfig.include,
        userConfig.exclude,
    );
    const timeout = userConfig.timeout || 10000;

    const _config: IConfig = {
        runner: {
            filesWithTests,
            timeout,
            onlyShowFiles: userConfig.onlyShowFiles || onlyShowFiles || false,
            args: userConfig.args || [],
        },
    };

    if (userConfig.clickhouse) {
        _config.clickhouse = {
            host: userConfig.clickhouse.host,
            port: userConfig.clickhouse.port,
            username: userConfig.clickhouse.username,
            password: userConfig.clickhouse.password,
            serviceName: userConfig.clickhouse.serviceName,
            version: userConfig.clickhouse.version,
            dbName: userConfig.clickhouse.dbName,
            tableName: userConfig.clickhouse.tableName,
        };
    }

    return _config;
}

export function findFilesWithTests(
    include: TSource,
    exclude?: TSource,
): TFilesWithTests {
    return fglob.sync(toArray(include), {
        concurrency: 1,
        ignore: toArray(exclude),
        absolute: true,
    });
}

export function toArray(source?: TSource): string[] {
    if (source instanceof Array) {
        return source;
    }

    if (typeof source === 'string') {
        return [source];
    }

    return [];
}
