import pMap from 'p-map';
import {URL} from 'url';

import {Runner, ARunner} from './baseRunner';
import {PuppeteerHelper} from '../PuppeteerHelper/puppeteerHelper';
import {BaseTest} from '../Test/baseTest';
import {ICoveredPage} from '../Test/ITest';
import {IGlobal, TCover, TTestCallback, IRunnerConfig} from './IRunner';

const requireAndForget = require('require-and-forget');

declare const global: IGlobal;

function createTest(
    name: string,
    uri: string,
    callback?: TTestCallback,
): BaseTest {
    return new BaseTest(name, uri, callback);
}

export class ChromeRunner extends Runner implements ARunner {
    private queue: BaseTest[] = [];
    private puppeteerHelper: PuppeteerHelper;

    constructor(config: IRunnerConfig) {
        super(config);

        this.puppeteerHelper = new PuppeteerHelper({
            // headless: false,
            // defaultViewport: null,
            ignoreHTTPSErrors: true,
            args: config.args,
        });

        global.cover = this.cover;
        config.filesWithTests.forEach(requireAndForget);
    }

    cover: TCover = (name, uri, callback) => {
        this.queue.push(createTest(name, uri, callback));
    };

    async run(): Promise<ICoveredPage[]> {
        await this.puppeteerHelper.init();

        const result = await pMap(this.queue, this.testMapper, {
            concurrency: 1,
        });

        await this.puppeteerHelper.deinit();

        return result;
    }

    private testMapper = async (test: BaseTest): Promise<ICoveredPage> => {
        const page = this.puppeteerHelper.getPage();
        const result = await test.run(page);
        return this.removeInlineResources(result);
    };

    private removeInlineResources(page: ICoveredPage): ICoveredPage {
        const pageUri = new URL(page.pageUri);

        const resources = page.resources.filter(resource => {
            const resourceUrl = new URL(resource.resourceUri);
            const isInlineMedia = resource.resourceUri.startsWith('data');
            const isInlineResource =
                resourceUrl.toString() === pageUri.toString();
            return !isInlineMedia && !isInlineResource;
        });

        return {...page, resources};
    }
}
