import {Command} from 'commander';
import colors from 'colors/safe';

import {ILighthouseScoreAndMetrics} from './types';

import getAverageScoreAndMetrics from './utilities/getAverageScoreAndMetrics';
import getCompareConsoleColor from './utilities/getCompareConsoleColor';

const program = new Command();

program
    .option(
        '--first-url <string>',
        'url to compare',
        'https://travel.yandex.ru/avia/',
    )
    .option(
        '--second-url <string>',
        'url to compare',
        'https://travel.yandex.ru/hotels/',
    )
    .option('-i, --iterations <number>', 'runs count', '10')
    .option('-d, --desktop', 'run as desktop')
    .option('-r, --report', 'need to generate report');

program.parse(process.argv);

const options = program.opts();

/**
 * Необходима минимум 12 версия nodejs.
 *
 * nvm use 12
 */
(async (): Promise<void> => {
    console.log(colors.green(`\nLighthouse comparison`));
    console.log(`\nFirst url: ${options.firstUrl}`);
    console.log(`Second url: ${options.secondUrl}`);
    console.log(`Iterations: ${options.iterations}`);

    const commonSettings = {
        iterations: Number(options.iterations),
        isDesktop: options.desktop,
        needToGenerateReport: options.report,
    };

    const firstResult = await getAverageScoreAndMetrics({
        url: options.firstUrl,
        ...commonSettings,
    });
    const secondResult = await getAverageScoreAndMetrics({
        url: options.secondUrl,
        ...commonSettings,
    });

    console.log('\nMetrics diff:');

    (
        Object.entries(firstResult) as [
            keyof ILighthouseScoreAndMetrics,
            number,
        ][]
    ).forEach(([metricKey, firstMetricValue]) => {
        const secondMetricValue = secondResult[metricKey];

        const metricDiff = Math.abs(secondMetricValue - firstMetricValue);
        const metricPercentageDiff = metricDiff
            ? Math.floor((metricDiff / firstMetricValue) * 100)
            : 0;

        const color = getCompareConsoleColor(
            metricKey,
            firstMetricValue,
            secondMetricValue,
        );

        console.log(
            color(
                `${metricKey}: ${firstMetricValue.toFixed(
                    2,
                )} -> ${secondMetricValue.toFixed(
                    2,
                )}, ${metricPercentageDiff}%`,
            ),
        );
    });
})();
