import moment from 'moment';
import ClickHouseApiClient from 'src/services/clickhouse/ClickHouseApiClient';

import {LIGHTHOUSE_RELEASE_METRICS_TABLE_COLUMNS} from './constants';
import {CLICKHOUSE_DATE_FORMAT} from 'src/services/clickhouse/constants';

import {IClickHouseLighthouseMeasure} from './types';
import {ILighthouseScoreAndMetrics} from '../types';

import getTestingVersion from './utilities/getTestingVersion';
import getPagesMetrics from '../utilities/getPagesMetrics/getPagesMetrics';
import hasAlreadyMeasured from './utilities/hasAlreadyMeasured';
import notifyToReleaseIssue from './utilities/notifyToReleaseIssue';
import getEnvVariable, {
    EEnvironmentVariables,
} from 'src/utilities/getEnvVariable';

const DB_NAME = 'db_lighthouse';
const TABLE_NAME = 'releaseMetrics';

(async (): Promise<void> => {
    try {
        const clickhouseConfig = {
            username: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CLICKHOUSE_USERNAME,
            ),
            password: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CLICKHOUSE_PASSWORD,
            ),
            host: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CLICKHOUSE_HOST,
            ),
            port: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CLICKHOUSE_PORT,
            ),
        };

        const lighthouseConfig = {
            configFile: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CONFIG_FILE,
            ),
            iterationsPerPage: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_ITERATIONS_PER_PAGE,
            ),
            concurrentMeasures: getEnvVariable(
                EEnvironmentVariables.LIGHTHOUSE_CONCURRENT_MEASURES,
            ),
        };

        const pagesList = require(lighthouseConfig.configFile).default;

        const lighthouseClickHouseClient =
            new ClickHouseApiClient<IClickHouseLighthouseMeasure>({
                ...clickhouseConfig,
                dbName: DB_NAME,
                tableName: TABLE_NAME,
                columns: LIGHTHOUSE_RELEASE_METRICS_TABLE_COLUMNS,
            });

        const testingVersion = await getTestingVersion();

        if (
            await hasAlreadyMeasured(testingVersion, lighthouseClickHouseClient)
        ) {
            console.log('Lighthouse: эта версия тестинга уже замерялась');

            return;
        }

        const startMeasuringTime = moment();

        console.log('Lighthouse: запуск замеров');

        const pagesMetrics = await getPagesMetrics(
            pagesList,
            Number(lighthouseConfig.concurrentMeasures),
            Number(lighthouseConfig.iterationsPerPage),
        );

        console.log(
            `Lighthouse: замеры заняли минут: ${moment().diff(
                startMeasuringTime,
                'minutes',
            )} (страниц: ${
                Object.keys(pagesList).length
            }, замеров на страницу: ${Number(
                lighthouseConfig.iterationsPerPage,
            )},
            параллельных потоков: ${Number(
                lighthouseConfig.concurrentMeasures,
            )})`,
        );

        const dateUtc = moment().utc().format(CLICKHOUSE_DATE_FORMAT);

        const metrics = (
            Object.entries(pagesMetrics) as [
                'string',
                ILighthouseScoreAndMetrics,
            ][]
        ).map(([pageId, pageMetrics]) => ({
            date: dateUtc,
            platform: 'mobile',
            version: testingVersion,
            page: pageId,
            score: Math.round(pageMetrics.score),
            fcp: Math.round(pageMetrics.fcp),
            lcp: Math.round(pageMetrics.lcp),
            si: Math.round(pageMetrics.speedIndex),
            tti: Math.round(pageMetrics.tti),
            tbt: Math.round(pageMetrics.tbt),
            cls: Math.floor(pageMetrics.cls * 100),
            mpf: Math.round(pageMetrics.maxPotentialFid),
            ttfb: Math.round(pageMetrics.ttfb),
            mtw: Math.round(pageMetrics.mainThreadWork),
            jset: Math.round(pageMetrics.jsExecutionTime),
            ds: Math.round(pageMetrics.domSize),
        }));

        if (
            await hasAlreadyMeasured(testingVersion, lighthouseClickHouseClient)
        ) {
            console.log('Lighthouse: эта версия тестинга уже замерялась');

            return;
        }

        try {
            console.log('Lighthouse: запуск нотификации в релизный тикет');

            await notifyToReleaseIssue({
                lighthouseClickHouseClient,
                metrics,
                testingVersion,
                pagesList,
            });

            console.log('Lighthouse: сообщение отправлено в релизный тикет');
        } catch (err) {
            console.error(
                'Lighthouse: ошибка нотификации в релизный тикет',
                err,
            );
        }

        await lighthouseClickHouseClient.insertValues(metrics);

        console.log('Lighthouse: замеры успешно загружены');
    } catch (err) {
        console.error(
            'Lighthouse: ошибка подсчета и отправки lighthouse метрик релиза',
            err,
        );
    }
})();
