import ClickHouseApiClient from 'src/services/clickhouse/ClickHouseApiClient';
import StartrekService from 'src/services/startrek/startrek';

import {IClickHouseLighthouseMeasure} from '../types';

import getEnvVariable, {
    EEnvironmentVariables,
} from 'src/utilities/getEnvVariable';

const SCORE_THRESHOLD_DIFF = 5;

export default async function notifyToReleaseIssue({
    lighthouseClickHouseClient,
    metrics,
    testingVersion,
    pagesList,
}: {
    lighthouseClickHouseClient: ClickHouseApiClient<IClickHouseLighthouseMeasure>;
    metrics: IClickHouseLighthouseMeasure[];
    testingVersion: number;
    pagesList: Record<'string', 'string'>;
}): Promise<void> {
    const startrekToken = getEnvVariable(EEnvironmentVariables.STARTREK_TOKEN);

    const prevReleaseMeasurements = await lighthouseClickHouseClient.getValues({
        limit: Object.values(pagesList).length,
        order: {
            by: 'date',
            sort: 'DESC',
        },
    });

    let message = `====Lighthouse мониторинг\nВерсия тестинга: ${testingVersion}\n`;

    let somePageScoreDecreased = false;
    let somePageScoreIncreased = false;

    metrics.forEach(metric => {
        const prevMetric = prevReleaseMeasurements.find(
            ({page}) => page === metric.page,
        );

        if (!prevMetric) {
            return;
        }

        const scoreDecline = Math.round(prevMetric.score - metric.score);

        if (Math.abs(scoreDecline) >= SCORE_THRESHOLD_DIFF) {
            const pageLink = `((${pagesList[metric.page]} ${metric.page}))`;

            message += `\n${pageLink}: оценка ${
                scoreDecline >= SCORE_THRESHOLD_DIFF
                    ? 'ухудшилась'
                    : 'улучшилась'
            } ${prevMetric.score} -> ${metric.score}`;

            if (scoreDecline >= SCORE_THRESHOLD_DIFF) {
                somePageScoreDecreased = true;
            } else {
                somePageScoreIncreased = true;
            }
        }
    });

    if (somePageScoreDecreased) {
        message += `\n\nОценка ${
            somePageScoreIncreased ? 'некоторых ' : ''
        }страниц выше значительно ухудшилась, что может свидетельствовать о снижении качества кода в релизе :(\nДежурный, попробуй разобраться в причине.`;
    } else if (somePageScoreIncreased) {
        message +=
            '\n\nОценка страниц выше значительно улучшилась. Так держать!';
    } else {
        message +=
            '\n\nОценка страниц в текущем релизе в пределах нормы. Можно работать дальше.';
    }

    message += `\n\n((https://datalens.yandex-team.ru/preview/g3vknm1le8n69-releasemetricschart?environment=production График)) ((https://a.yandex-team.ru/svn/trunk/arcadia/travel/frontend/portal/docs/performance/releaseMonitoring.md Документация))`;

    const releaseIssue = getEnvVariable(
        EEnvironmentVariables.STARTREK_TICKET_ID,
    );

    const STClient = new StartrekService({
        oauth: startrekToken,
    });

    await STClient.appendComment({
        ticket: releaseIssue,
        text: message,
    });
}
