import mean from 'lodash/mean';
import * as chromeLauncher from 'chrome-launcher';

import {
    ILighthouseCalculateOptions,
    ILighthouseScoreAndMetrics,
} from '../types';

import getScoreAndMetrics from './getScoreAndMetrics';

export default async function getAverageScoreAndMetrics({
    url,
    iterations,
    isDesktop = false,
    needToGenerateReport = false,
    withLogs = false,
}: ILighthouseCalculateOptions): Promise<ILighthouseScoreAndMetrics> {
    const metricValuesMap = new Map<keyof ILighthouseScoreAndMetrics, number[]>(
        [],
    );

    if (withLogs) {
        console.log(`\n${url}`);
    }

    const chrome = await chromeLauncher.launch({
        chromeFlags: ['--headless', '--ignore-certificate-errors'],
    });

    for (let i = 0; i < iterations; i++) {
        const results = await getScoreAndMetrics({
            url,
            chrome,
            isDesktop,
            needToGenerateReport,
            reportFileName: `${url.replace(/\//g, '-')}-${i}`,
        });

        if (withLogs) {
            console.log(`\n${url} ${i + 1}`, Math.round(results.score));
        }

        (
            Object.entries(results) as [
                keyof ILighthouseScoreAndMetrics,
                number,
            ][]
        ).forEach(([metricKey, metricValue]) => {
            const metricValues = metricValuesMap.get(metricKey);

            if (metricValues) {
                metricValues.push(metricValue);
            } else {
                metricValuesMap.set(metricKey, [metricValue]);
            }
        });
    }

    await chrome.kill();

    return [...metricValuesMap.entries()].reduce(
        (accMetrics, [metricKey, metricValue]) => {
            accMetrics[metricKey] = mean(metricValue);

            return accMetrics;
        },
        {} as ILighthouseScoreAndMetrics,
    );
}
