import fs from 'fs-extra';
import {LaunchedChrome} from 'chrome-launcher';
// @ts-ignore
import lighthouse from 'lighthouse';

// @ts-ignore
import lighthouseConstants from 'lighthouse/lighthouse-core/config/constants';

import {
    ILighthouseCalculateOptions,
    ILighthouseScoreAndMetrics,
} from '../types';

export default async function getScoreAndMetrics({
    url,
    chrome,
    isDesktop = false,
    needToGenerateReport = false,
    reportFileName = 'report',
}: Omit<ILighthouseCalculateOptions, 'iterations'> & {
    chrome: LaunchedChrome;
    reportFileName?: string;
}): Promise<ILighthouseScoreAndMetrics> {
    const {
        report,
        lhr: {categories, audits},
    } = await lighthouse(url, {
        logLevel: 'silent',
        output: needToGenerateReport ? 'html' : 'json',
        onlyCategories: ['performance'],
        // Указание хоста в таком формате форсит использовать IPv4, из-за каких-то особенностей
        // Sandbox, LH не мог подключиться по IPv6
        hostname: '0.0.0.0',
        port: chrome.port,
        ...(isDesktop
            ? {
                  formFactor: 'desktop',
                  throttling: lighthouseConstants.throttling.desktopDense4G,
                  screenEmulation:
                      lighthouseConstants.screenEmulationMetrics.desktop,
                  emulatedUserAgent: lighthouseConstants.userAgents.desktop,
              }
            : {}),
    });

    if (needToGenerateReport) {
        await fs.ensureDir('tools/lighthouse/reports');
        await fs.writeFile(
            `tools/lighthouse/reports/${reportFileName}.html`,
            report,
        );
    }

    return {
        score: categories.performance.score * 100,
        fcp: audits['first-contentful-paint'].numericValue,
        lcp: audits['largest-contentful-paint'].numericValue,
        speedIndex: audits['speed-index'].numericValue,
        tti: audits.interactive.numericValue,
        tbt: audits['total-blocking-time'].numericValue,
        cls: audits['cumulative-layout-shift'].numericValue,
        maxPotentialFid: audits['max-potential-fid'].numericValue,
        ttfb: audits['server-response-time'].numericValue,
        mainThreadWork: audits['mainthread-work-breakdown'].numericValue,
        jsExecutionTime: audits['bootup-time'].numericValue,
        domSize: audits['dom-size'].numericValue,
    };
}
