import fs from 'fs';
import StartrekService from 'src/services/startrek/startrek';

import {Logger} from 'src/utilities/logger';
import getEnvVariable, {
    EEnvironmentVariables,
} from 'src/utilities/getEnvVariable';

interface IHermioneSingleTestReport {
    status: 'success' | 'fail' | 'skipped';
    [key: string]: unknown;
}

(async function () {
    const logger = new Logger('E2E_ST_NOTIFY');

    try {
        const ticket = getEnvVariable(
            EEnvironmentVariables.STARTREK_TICKET_ID,
            '',
        );

        if (!ticket) {
            return;
        }

        const startrekToken = getEnvVariable(
            EEnvironmentVariables.STARTREK_TOKEN,
        );
        const flowLink = getEnvVariable(EEnvironmentVariables.CI_FLOW_LINK);
        const enableFilterFallenTests =
            getEnvVariable(
                EEnvironmentVariables.E2E_ENABLE_FILTER_FALLEN_TESTS,
                'false',
            ) === 'true';
        const reportPath = getEnvVariable(
            EEnvironmentVariables.E2E_JSON_REPORT_PATH,
        );

        const reportRaw = fs.readFileSync(reportPath, 'utf8');
        const report = JSON.parse(reportRaw);

        const summary = (
            Object.values(report) as IHermioneSingleTestReport[]
        ).reduce(
            (acc, test) => {
                acc[test.status] = acc[test.status] + 1;

                return acc;
            },
            {
                success: 0,
                fail: 0,
                skipped: 0,
            },
        );

        const {success, fail, skipped} = summary;

        const message = [
            `Результаты тестов (${
                enableFilterFallenTests ? 'только упавших' : 'всех'
            }):`,
            success && `!!(green)✓ прошло ${success}!!`,
            fail && `!!(red)✗ упало ${fail}!!`,
            skipped && `!!(grey)✗ в игноре ${skipped}!!`,
            ' ',
            flowLink && `((${flowLink} CI Flow))`,
        ]
            .filter(Boolean)
            .join('\n');

        const STClient = new StartrekService({
            oauth: startrekToken,
        });

        await STClient.appendComment({
            ticket,
            text: message,
        });
    } catch (error) {
        if (error instanceof Error) {
            logger.log(error.message);
        }
    } finally {
        logger.success('ok');
    }
})();
