import path from 'path';
import fs from 'fs-extra';
import moment from 'moment';
/** @deprecated
 *  Нужно зарефаторить ClickhouseApiClient в отдельный npm-пакет и заиспользовать
 *  его в других местах (bundlemeter, lighthouse…)
 * **/
import ClickHouseApiClient from 'src/services/clickhouse/ClickHouseApiClient';

import {E2E_CLICKHOUSE_COLUMNS} from './constants/clickhouse';
import {BUILD_TYPE_RELEASE, TEAMCITY_DATE_FORMAT} from './constants/teamcity';
import {CLICKHOUSE_DATE_FORMAT} from 'src/services/clickhouse/constants';

import {IClickhouseReleaseRun} from './types/clickhouse';
import {THermioneTestsReport} from './types/hermione';

import TeamcityApiClient from '../../../../../portal/tools/utils/api/TeamcityApiClient/TeamcityApiClient';
import {getAllFailedTriesCount} from './utils/getAllFailedTriesCount';
import getEnvVariable, {
    EEnvironmentVariables,
} from 'src/utilities/getEnvVariable';

const DB_NAME = 'db_e2e_test_metrics';
const TABLE_NAME = 'e2e_release_runs';

const JSON_REPORT_PATH = './integration/json-report/report.json';

require('../utils/prepareEnvVars')();

(async (): Promise<void> => {
    const clickhouseUsername = getEnvVariable(
        EEnvironmentVariables.E2E_CLICKHOUSE_USERNAME,
    );
    const clickhousePassword = getEnvVariable(
        EEnvironmentVariables.E2E_CLICKHOUSE_PASSWORD,
    );
    const clickhouseHost = getEnvVariable(
        EEnvironmentVariables.E2E_CLICKHOUSE_PASSWORD,
    );
    const clickhousePort = getEnvVariable(
        EEnvironmentVariables.E2E_CLICKHOUSE_PORT,
    );

    if (!clickhouseUsername || !clickhousePassword) {
        throw new Error(
            'Требуются CLICKHOUSE_E2E_USERNAME и CLICKHOUSE_E2E_PASSWORD для авторизации в ClickHouse',
        );
    }

    const clickhouseClient = new ClickHouseApiClient<IClickhouseReleaseRun>({
        dbName: DB_NAME,
        host: clickhouseHost,
        port: clickhousePort,
        tableName: TABLE_NAME,
        username: clickhouseUsername,
        password: clickhousePassword,
        columns: E2E_CLICKHOUSE_COLUMNS,
    });

    const teamcityClient = new TeamcityApiClient();

    const branchName = process.env.BRANCH_NAME ?? '';
    const releaseTicketName = branchName.match(/TRAVELFRONT-\d+/)?.[0];

    if (!releaseTicketName) {
        throw new Error(
            `Неверное название релизного тикета для ветки ${branchName}`,
        );
    }

    const builds = await teamcityClient.getBuilds({
        buildType: BUILD_TYPE_RELEASE,
        count: 1e4,
        state: 'running',
        branch: {
            policy: 'ALL_BRANCHES',
        },
    });

    const branchBuild = builds.find(
        ({branchName: buildBranchName}) => buildBranchName === branchName,
    );

    if (!branchBuild) {
        throw new Error('Не найден запущенный билд релизных тестов');
    }

    const detailedBuild = await teamcityClient.getBuild(branchBuild.id);
    const jsonReport: THermioneTestsReport = await fs.readJson(
        path.resolve(JSON_REPORT_PATH),
    );

    const duration =
        Date.now() -
        Number(moment(detailedBuild.startDate, TEAMCITY_DATE_FORMAT));
    const passedCount = detailedBuild.testOccurrences?.passed ?? 0;
    const failedCount = detailedBuild.testOccurrences?.failed ?? 0;
    const ignoredCount = detailedBuild.testOccurrences?.ignored ?? 0;
    const retriesCount = getAllFailedTriesCount(jsonReport) - failedCount;

    const whereQuery = `ticketName='${releaseTicketName}'`;
    const [currentReleaseRun] = await clickhouseClient.getValues({
        where: whereQuery,
    });

    const runValues = {
        date: moment().utc().format(CLICKHOUSE_DATE_FORMAT),
        passedCount,
        failedCount,
        ignoredCount,
        retriesCount,
        duration,
    };

    if (currentReleaseRun) {
        await clickhouseClient.updateValue(runValues, whereQuery);
    } else {
        await clickhouseClient.insertValue({
            ticketName: releaseTicketName,
            ...runValues,
        });
    }
})().catch(err => {
    console.log(err);

    process.exit(1);
});
