import {scissorsComment} from '../constants/common';

import {ArcanumService} from '../services/arcanum/arcanum';

interface IItem {
    key: string;
    value: string;
}

function getDescriptionWithCITable(
    description: string,
    items: IItem[],
): string {
    const startLabel = '<!-- CI start -->';
    const endLabel = '<!-- CI end -->';

    const createTableContent = (_items: IItem[]) => {
        return [
            `${startLabel}`,
            `| CI | |`,
            `| --- | --- |`,
            ..._items.map(({key, value}) => `| ${key} | ${value} |`),
            `${endLabel}`,
        ].join('\n');
    };

    const [tableContent] =
        description.match(new RegExp(`${startLabel}.*${endLabel}`, 's')) || [];

    let nextDescription: string;

    if (tableContent) {
        const foundIndexes: number[] = [];

        const rows = tableContent
            .split('\n')
            .slice(3, -1)
            .map((row: string) => {
                const matches = row.match(/\| ([^|]+) \| ([^|]+) \|/s) || [];
                const rowKey = matches[1];
                const rowValue = matches[2];
                const foundIndex = items.findIndex(({key}) => key === rowKey);

                foundIndexes.push(foundIndex);

                return {
                    key: rowKey,
                    value:
                        foundIndex > -1
                            ? String(items[foundIndex].value)
                            : rowValue,
                };
            })
            .concat(items.filter((_, index) => !foundIndexes.includes(index)));

        nextDescription = description.replace(
            tableContent,
            createTableContent(rows),
        );
    } else {
        nextDescription = [
            description,
            description.includes(scissorsComment) ? '' : scissorsComment,
            '',
            createTableContent(items),
        ].join('\n');
    }

    return nextDescription;
}

export async function changePullRequestCITable(
    pullRequestId: string,
    items: IItem[],
    onChangeDescription?: (description: string) => string,
): Promise<void> {
    const arcanum = new ArcanumService();

    const {data} = await arcanum.getPullRequest(pullRequestId, 'description');
    const description: string = data.data.description || '';

    let nextDescription = getDescriptionWithCITable(description, items);

    if (onChangeDescription) {
        nextDescription = onChangeDescription(nextDescription);
    }

    if (description !== nextDescription) {
        await arcanum.changeDescription(pullRequestId, nextDescription);
    }
}
