import {dependenciesKeys} from 'src/constants/common';

import {IBadge} from 'src/types/common';
import {IExecOptions} from 'src/types/process';

import {isArcCommand} from 'src/utilities/arc';
import {appendToFile, readFileJson} from 'src/utilities/fs';
import {isNpmCommand} from 'src/utilities/npm';
import {execArc} from './arc.helpers';
import {execNpm} from './npm.helpers';
import {Process} from 'src/utilities/process';

export async function addResource(
    resource: string | undefined,
    path = 'common',
) {
    await appendToFile(
        `${process.env.RESULT_RESOURCES_PATH}/${path}`,
        resource ? `${resource}\n` : '',
    );
}

export async function addBadge(badge: IBadge | undefined) {
    await appendToFile(
        `${process.env.RESULT_BADGES_PATH}/common`,
        badge ? `${JSON.stringify(badge)}\n` : '',
    );
}

export async function initResource(path?: string) {
    await addResource(undefined, path);
}

export async function initBadge() {
    await addBadge(undefined);
}

export async function addPrBadge(prUrl?: string) {
    let badge: IBadge | undefined;

    if (prUrl) {
        badge = {
            id: 'pr',
            module: 'ARCADIA',
            status: 'SUCCESSFUL',
            text: 'Pull request url',
            url: prUrl,
        };
    }

    await addBadge(badge);
}

export async function getPackage(packageName: string, packageJsonPath: string) {
    const packageJson = await readFileJson(packageJsonPath);

    for (const dependencyKey of dependenciesKeys) {
        const dependencies = packageJson[dependencyKey] as Record<
            string,
            string
        >;
        const version: string | undefined = dependencies[packageName];

        if (version) {
            return {dependencyKey, version};
        }
    }

    return {};
}

function prepareCommand(command: string) {
    return command
        .split('\n')
        .reduce((commands, row) => {
            const _command = row.trim();

            if (_command) {
                commands.push(_command);
            }

            return commands;
        }, [] as string[])
        .join(' && ');
}

export async function exec(input: string | string[], options?: IExecOptions) {
    const cmd = new Process();
    const commands = Array.isArray(input) ? input : input.split(';;');
    const results: {stdout: string; stderr: string}[] = [];
    const customExecutors = [
        {check: isArcCommand, executor: execArc},
        {check: isNpmCommand, executor: execNpm},
    ];

    for (const command of commands) {
        const execCommand = prepareCommand(command);

        if (execCommand) {
            const {executor} =
                customExecutors.find(({check}) => check(execCommand)) || {};
            const result = executor
                ? await executor(execCommand, options)
                : await cmd.exec(execCommand, options);

            results.push(result);
        }
    }

    return results;
}
