import {dependenciesKeys, devDependencyKey} from '../constants/common';

import {IExecOptions} from 'src/types/process';

import {createBranch} from 'src/utilities/arc';
import {isNpmCommand, isNpmError} from 'src/utilities/npm';
import {getPackage} from './common.helpers';
import {Process} from 'src/utilities/process';

export async function execNpm(command: string, options?: IExecOptions) {
    const cmd = new Process();
    const npmCommand = isNpmCommand(command) ? command : `npm ${command}`;

    return await cmd.exec(npmCommand, {...options, isError: isNpmError});
}

export async function preparePackageInstall(
    props: {
        repoPath?: string;
        packageName?: string;
        installAs?: string;
        version?: string;
        targetBranch?: string;
        ARCADIA_PATH?: string;
    } & NodeJS.ProcessEnv,
) {
    const {
        repoPath: repoDir,
        packageName,
        installAs,
        ARCADIA_PATH: arcadiaPath,
    } = props;
    const version = props.version || 'latest';
    const targetBranch = props.targetBranch || '*';

    if (!repoDir || !packageName) {
        throw Error('Repo path and package must be specified.');
    }

    if (
        installAs &&
        !dependenciesKeys.includes(installAs as typeof dependenciesKeys[number])
    ) {
        throw Error(
            `"Install as" value must be one of "${dependenciesKeys.join(
                ', ',
            )}".`,
        );
    }

    const repoPath = `${arcadiaPath}/${repoDir}`;
    const repoDirLast = repoDir.replace(/^.*\//, '');
    const packageNameLast = packageName.replace(/^.*\//, '');
    const packageJsonPath = `${repoPath}/package.json`;

    const dependencyKey =
        installAs ||
        (await getPackage(packageName, packageJsonPath)).dependencyKey;
    const installArgs = dependencyKey === devDependencyKey ? '--save-dev' : '';

    const command = `npm install ${installArgs} ${packageName}@${version}`;

    const targetVersion =
        (
            await execNpm(`npm view ${packageName} dist-tags.${version}`, {
                silent: true,
            })
        ).stdout || version;

    let branch: string;

    if (targetBranch.endsWith('*')) {
        branch = createBranch([
            targetBranch.slice(0, -1),
            repoDirLast,
            packageNameLast,
            /** Cut prerelease or alpha/beta versions (for 2.1.0-1 or 2.1.0-beta.1 will be 2.1.0) */
            targetVersion.replace(/-[^-]*$/, ''),
        ]);
    } else {
        branch = targetBranch;
    }

    const commitMessage = [
        `chore: install ${packageName}@${targetVersion} in ${repoDirLast} [no-changelog]`,
        `Repo path: \`${repoDir}\``,
        `Package name: \`${packageName}\``,
        `Version: \`${version}\``,
        `Install as: \`${dependencyKey || '-'}\``,
    ].join('\n');

    return {
        targetBranch: branch,
        command,
        commitMessage,
    };
}
