import {IAxiosRequestConfig, ILoggerInterface} from 'src/types/common';

import {request} from 'src/utilities/request';
import {Logger} from 'src/utilities/logger';

interface IArcanumRequestConfig extends Omit<IAxiosRequestConfig, 'url'> {
    path?: string;
}

interface IArcanumResponse<T = unknown> {
    data: T;
    errors: {
        status: string;
        message: string;
        details: Record<string, unknown>;
    }[];
}

interface IPullRequest {
    description?: string;
}

interface IDiffSet {
    id?: string;
    status?: string;
    published_at?: string;
}

export class ArcanumService {
    private logger: ILoggerInterface;
    private url = 'https://a.yandex-team.ru/api/v1';

    constructor() {
        this.logger = new Logger('ArcanumService');
    }

    postComment(prId: string, content: string) {
        return this.request({
            path: `/review-requests/${prId}/comments`,
            method: 'post',
            data: {content},
        });
    }

    getComments(prId: string) {
        return this.request({path: `/review-requests/${prId}/comments`});
    }

    getPullRequest(prId: string, fields?: string) {
        return this.request<IArcanumResponse<IPullRequest>>({
            path: `/pull-requests/${prId}`,
            ...(fields ? {params: {fields}} : {}),
        });
    }

    getDiffSets(prId: string, fields?: string) {
        return this.request<IArcanumResponse<IDiffSet[]>>({
            path: `/review-requests/${prId}/diff-sets`,
            ...(fields ? {params: {fields}} : {}),
        });
    }

    changeDescription(prId: string, description: string) {
        return this.request({
            path: `/review-requests/${prId}/description`,
            method: 'put',
            data: {description},
        });
    }

    blockMerge(method: 'post' | 'delete', prId: string, diffSetId: string) {
        return this.request({
            path: `/review-requests/${prId}/diff-sets/${diffSetId}/block-merge`,
            method,
        });
    }

    private request<T>({
        method = 'get',
        path,
        ...axiosConfig
    }: IArcanumRequestConfig) {
        return request<T>({
            method,
            ...axiosConfig,
            url: `${this.url}${path}`,
            headers: {
                Authorization: `OAuth ${process.env.ARCANUM_TOKEN}`,
                'Content-Type': 'application/json',
                ...axiosConfig.headers,
            },
            retries: 1,
            logger: this.logger,
        });
    }
}
