import {ILoggerInterface} from 'src/types/common';
import {EInstanceStatus, IGetInstanceResponse, TFarmContext} from './types';

import {request} from 'src/utilities/request';
import {Logger} from 'src/utilities/logger';
import {delay} from 'src/utilities/delay';

export default class FarmService {
    private static getStatus(
        prevStatus: EInstanceStatus,
        nextStatus: EInstanceStatus,
    ) {
        return nextStatus === EInstanceStatus.RUNNING &&
            prevStatus !== EInstanceStatus.GENERATING
            ? EInstanceStatus.GENERATING
            : nextStatus;
    }

    private logger: ILoggerInterface;
    private context: TFarmContext;

    constructor(context: TFarmContext) {
        this.logger = new Logger('FarmService');
        this.context = context;
    }

    getProjectUrl() {
        return `${this.context.farmUrl}/project/${this.context.project}`;
    }

    webhook({action}: {action: 'opened' | 'closed'}) {
        const {
            project,
            organization,
            pullRequestId,
            pullRequestSummary,
            featureBranch,
            upstreamBranch,
            farmUrl,
            ...restContext
        } = this.context;

        const env = Object.entries(restContext).reduce<
            Record<string, string | undefined>
        >((accum, [key, value]) => {
            if (key.startsWith('env_')) {
                accum[key.replace('env_', '')] = value;
            }

            return accum;
        }, {});

        return request({
            method: 'post',
            url: `${farmUrl}/webhook`,
            data: {
                action,
                context: {
                    project,
                    organization,
                    pull_request_id: pullRequestId,
                    pull_request_summary: pullRequestSummary,
                    feature_branch: featureBranch,
                    upstream_branch: upstreamBranch,
                },
                env,
            },
            headers: {'x-arcanum-ci-event': 'pull_request'},
            logger: this.logger,
        });
    }

    getInstance() {
        const {project, featureBranch, farmUrl} = this.context;

        return request<IGetInstanceResponse>({
            method: 'post',
            url: `${farmUrl}/api/getInstance`,
            data: {
                vcs: 'arc',
                project,
                branch: featureBranch,
            },
        });
    }

    async pollInstance() {
        let i = 0;
        let timeout = 0;
        let data = {} as IGetInstanceResponse;
        let status: EInstanceStatus = EInstanceStatus.QUEUED;

        while (
            [EInstanceStatus.QUEUED, EInstanceStatus.GENERATING].includes(
                status,
            )
        ) {
            await delay(timeout);

            const response = await this.getInstance();

            data = response.data;
            status = FarmService.getStatus(status, data.instance.status);

            this.logger?.log(`Iteration ${i}: ${status}`);

            timeout = timeout || Number(this.context.pollTimeout) || 30000;
            i++;
        }

        if (status !== EInstanceStatus.RUNNING) {
            throw new Error(
                `Instance ${data.instance.instanceName} is ${status}`,
            );
        }

        return data;
    }
}
