import axios, {AxiosInstance} from 'axios';

import {
    ICommentsResponse,
    IConstructorArg,
    ICreateCommentParams,
    IIssueLink,
    IPatchIssueParams,
    IStTicket,
    IIssueBaseParams,
    ITicketInfo,
    IAddChecklistParams,
    IStoredIssueChecklistItem,
} from './types';
import {ILoggerInterface} from 'src/types/common';

import {Logger} from 'src/utilities/logger';

export default class StartrekService {
    private logger: ILoggerInterface;
    private api: AxiosInstance;

    constructor({host, oauth}: IConstructorArg) {
        this.logger = new Logger('StartrekService');
        this.api = axios.create({
            baseURL: host || 'https://st-api.yandex-team.ru',
            headers: {
                authorization: `OAuth ${oauth}`,
            },
        });
    }

    async createTicket({
        queue,
        parent,
        assignee,
        summary,
        description,
        components,
        tags,
        links,
    }: ITicketInfo): Promise<IStTicket> {
        this.logger.log(
            'Creating ST ticket in %s, POST %s/v2/issues',
            queue,
            this.api.defaults.baseURL,
        );

        const reqParams = {
            queue,
            parent,
            assignee,
            summary,
            descriptionRenderType: 'WF',
            description,
            components,
            tags,
            links,
        };

        const result = await this.api.post<{key: string}>(
            '/v2/issues',
            reqParams,
        );

        return result.data;
    }

    async appendComment({ticket, text}: ICreateCommentParams) {
        if (!ticket) {
            throw Error('Ticket is required!');
        }

        this.logger.log(
            'Append ST comment in %s, POST %s/v2/issues',
            ticket,
            this.api.defaults.baseURL,
        );

        const result = await this.api.post<{key: string}>(
            `/v2/issues/${ticket}/comments?isAddToFollowers=false`,
            {text},
        );

        return result.data;
    }

    async getTicketCommentsList({ticket}: IIssueBaseParams) {
        const result: ICommentsResponse = await this.api.get(
            `/v2/issues/${ticket}/comments`,
        );
        const ticketComments = result.data || [];

        return ticketComments.map(({text}) => text);
    }

    async getIssueLinks({ticket}: IIssueBaseParams): Promise<IIssueLink[]> {
        const result = await this.api.get<IIssueLink[]>(
            `/v2/issues/${ticket}/links`,
        );

        return result.data || [];
    }

    async getChecklistItems({
        ticket,
    }: IIssueBaseParams): Promise<IStoredIssueChecklistItem[]> {
        const result = await this.api.get<IStoredIssueChecklistItem[]>(
            `/v2/issues/${ticket}/checklistItems`,
        );

        return result.data || [];
    }

    async addChecklistItems({
        ticket,
        items,
    }: IAddChecklistParams): Promise<void> {
        return await this.api.post(
            `/v2/issues/${ticket}/checklistItems`,
            items,
        );
    }

    async removeChecklistItems({ticket}: IIssueBaseParams): Promise<void> {
        return await this.api.delete(`/v2/issues/${ticket}/checklistItems`);
    }

    async patchIssue({ticket, fields}: IPatchIssueParams): Promise<void> {
        return await this.api.patch(`/v2/issues/${ticket}`, fields);
    }
}
