import {ExecException} from 'child_process';

import {IAxiosError, ILoggerInterface} from '../types/common';

export class Logger implements ILoggerInterface {
    private context?: string;

    constructor(context?: string) {
        this.context = context;
    }

    log = (message: string, ...args: unknown[]): void => {
        /** Log via console.error to prevent getting into stdout */
        console.error(this.getMessage(message), ...args);
    };

    success = (message: string, ...args: unknown[]): void => {
        console.log(this.getMessage(message), ...args);
    };

    error = (message: string, ...args: unknown[]): void => {
        console.error(this.getMessage(message), ...args);
    };

    logError = (error?: IAxiosError | Error | ExecException | null): void => {
        if (error) {
            if ('isAxiosError' in error) {
                this.error('Axios', error);
            } else {
                const {name, message, stack} = error;

                this.error(name);
                this.error(message);

                if (stack) {
                    this.error(stack);
                }
            }
        }
    };

    setContext(context: string): void {
        this.context = context;
    }

    toString(): string {
        return 'Logger';
    }

    private getMessage(message: string): string {
        const now = new Date();
        const time = now.toLocaleTimeString('ru-RU');

        return `[${time}][${this.context || ''}] ${message}`;
    }
}
