import * as childProcess from 'child_process';
import util from 'util';

import {ILoggerInterface} from '../types/common';
import {IExecOptions, IExecResult} from '../types/process';

import {Logger} from './logger';

const exec = util.promisify(childProcess.exec);

export class Process {
    private logger: ILoggerInterface;

    constructor() {
        this.logger = new Logger('Process');
    }

    async exec(
        command: string,
        options: IExecOptions = {},
    ): Promise<{stdout: string; stderr: string}> {
        this.logger.log(
            `Exec "${command}" with options ${JSON.stringify(options)}`,
        );

        const {isError, silent, env, ...restOptions} = options;
        const maxBuffer = Number(process.env.nodeProcessMaxBuffer);
        const minMaxBuffer = 1024 * 1024 * 100;

        let result: IExecResult;

        try {
            result = await exec(command, {
                maxBuffer: Number.isNaN(maxBuffer)
                    ? minMaxBuffer
                    : Math.max(maxBuffer, minMaxBuffer),
                env: {...process.env, ...env},
                ...restOptions,
            });
        } catch (error) {
            if (!silent) {
                throw error;
            }

            result = {stdout: '', stderr: ''};
        }

        const stdout = result.stdout.toString().trim();
        const stderr = result.stderr.toString().trim();

        this.logger.log(
            `stdout: ${stdout}${stderr ? `\nstderr: ${stderr}\n` : ''}`,
        );

        if (!silent && stderr && (!isError || isError(stderr))) {
            throw Error(`Error on command: "${command}"\nstderr: ${stderr}`);
        }

        return {stdout, stderr};
    }
}
