import {
    ILoggerInterface,
    IAxiosRequestConfig,
    IAxiosResponse,
} from '../types/common';

import {requireAxios} from '../helpers/require.helpers';

import {delay} from './delay';

const axios = requireAxios();

// eslint-disable-next-line @typescript-eslint/explicit-function-return-type
export function request<T = unknown>(requestConfig?: IAxiosRequestConfig) {
    const {retries = 3, logger, ...config} = requestConfig || {};

    // eslint-disable-next-line @typescript-eslint/explicit-function-return-type
    const callback = async () => {
        logger?.log('Request config:', config);

        const response = (await axios.request(config)) as IAxiosResponse<T>;

        logger?.log('Response data:', response.data);

        return response;
    };

    return retriable(callback, {retries, logger})();
}

// eslint-disable-next-line @typescript-eslint/explicit-function-return-type
export function retriable<T extends (...args: unknown[]) => Promise<unknown>>(
    callback: T,
    options?: {retries?: number; logger?: ILoggerInterface},
) {
    const {retries = 3, logger} = options || {};
    let counter = 0;

    const retriableCallback = async (...args: unknown[]): Promise<unknown> => {
        try {
            return await callback(...args);
        } catch (error) {
            if (counter >= retries) {
                throw error;
            } else {
                counter++;

                await delay(counter * 1000);

                logger?.log(`Retry ${counter}`);

                return await retriableCallback(...args);
            }
        }
    };

    return retriableCallback as T;
}
