#!/usr/bin/env node

import yargs from 'yargs';

import EExportMode, {EXPORT_MODES, isExportMode} from '../types/EExportMode';

import {resolveConfig} from './resolveConfig';
import run from '../app/app';
import generate from '../core/generate';

const FIGMA_AUTH_TOKEN = process.env.FIGMA_AUTH_TOKEN;

(async (): Promise<void> => {
    const config = await resolveConfig();

    const packageJson = require('../../package.json');

    const argv = await yargs
        .usage('Frigga — Figma react icon generator \n\nUsage: $0 [options]')
        .help('help')
        .alias('help', 'h')
        .version('version', packageJson.version)
        .alias('version', 'v')
        .options({
            output: {
                alias: 'o',
                describe: '<directory> output directory',
                requiresArg: true,
                demandOption: !config.outputDirectory,
                type: 'string',
            },
            file: {
                alias: 'f',
                describe: '<key> key of Figma file with icons',
                requiresArg: true,
                demandOption: !config.file,
                type: 'string',
            },
            token: {
                alias: 't',
                describe: '<token> Figma auth token',
                requiresArg: true,
                demandOption: !FIGMA_AUTH_TOKEN,
                type: 'string',
            },
            host: {
                describe: '<host> Figma api host',
                requiresArg: true,
                demandOption: false,
                type: 'string',
                default: 'api.figma.com',
            },
            cache: {
                describe: 'Check if icons is up to date',
                requiresArg: false,
                demandOption: false,
                type: 'boolean',
                default: true,
            },
            mode: {
                describe: EXPORT_MODES.map(v => `"${v}"`).join(' or '),
                choices: EXPORT_MODES,
                demandOption: false,
                default: EExportMode.REACT_COMPONENT,
            },
        }).argv;

    const authToken = argv.token || FIGMA_AUTH_TOKEN;

    if (!authToken) {
        throw new Error(
            'Set figma auth token: via CLI param --token or -t or via env variable FIGMA_AUTH_TOKEN',
        );
    }

    const exportMode = config.exportMode ?? argv.mode;

    const options = {
        ...config,
        file: config.file ?? argv.file,
        apiHost: config.apiHost ?? argv.host,
        authToken,
        outputDirectory: config.outputDirectory ?? argv.output,
        noCache: config.noCache ?? !argv.cache,
        exportMode: isExportMode(exportMode)
            ? exportMode
            : EExportMode.REACT_COMPONENT,
    };

    const [{hasErrors}] = await Promise.all([generate(options), run()]);

    if (hasErrors) {
        process.exit(1);
    }
})().catch(e => {
    console.error(e);

    process.exit(1);
});
