import {differenceBy} from 'lodash';
import {Client as createClient} from 'figma-js';
import {AxiosInstance} from 'axios';

import IFriggaOptions from '../types/IFriggaOptions';
import EExportMode from '../types/EExportMode';

import MetaFileProvider from './utilities/MetaFileProvider';
import filterComponentsWithChanges from './utilities/filterComponentsWithChanges/filterComponentsWithChanges';
import renderComponents from './utilities/renderComponents';
import processComponents from './utilities/processComponents/processComponents';
import collectMeta from './utilities/collectMeta';
import defaultGetComponentName from './utilities/defaultConfig/getComponentName';
import defaultGetSvgrConfig from './utilities/defaultConfig/getSvgrConfig';
import defaultGetSvgoConfig from './utilities/defaultConfig/getSvgoConfig';
import createDefaultGetFileName from './utilities/defaultConfig/createGetFileName';
import setAxiosRetry from './utilities/setAxiosRetry';

import {EEventType, emit} from '../eventEmitter';

interface IGenerateResult {
    hasErrors: boolean;
}

export default async function generate({
    apiHost,
    authToken,
    exportMode = EExportMode.REACT_COMPONENT,
    file,
    outputDirectory,
    select,
    getSvgoConfig = defaultGetSvgoConfig,
    getSvgrConfig = defaultGetSvgrConfig,
    getComponentName = defaultGetComponentName,
    getFileName = createDefaultGetFileName(exportMode),
    noCache,
    exportScale,
    exportPngTemplate,
}: IFriggaOptions): Promise<IGenerateResult> {
    const metaFileProvider = new MetaFileProvider({outputDirectory, noCache});
    const figmaClient = createClient({
        apiRoot: apiHost,
        personalAccessToken: authToken,
    });

    setAxiosRetry(figmaClient.client as AxiosInstance);

    emit(EEventType.FILE_COMPONENTS_REQUEST);

    const res = await figmaClient.fileComponents(file);
    const components = res.data.meta.components.filter(iconNode =>
        select(iconNode),
    );

    emit(EEventType.FILE_COMPONENTS_SUCCESS, components);

    emit(EEventType.FILTER_COMPONENTS_REQUEST);

    const prevMeta = await metaFileProvider.read();
    const componentsWithChanges = filterComponentsWithChanges(
        components,
        prevMeta,
    );

    emit(EEventType.FILTER_COMPONENTS_SUCCESS);

    emit(EEventType.RENDER_COMPONENTS_REQUEST);

    const imageUrls = await renderComponents({
        figmaClient,
        components: componentsWithChanges,
        exportMode,
        exportScale,
    });

    emit(EEventType.RENDER_COMPONENTS_SUCCESS);

    emit(EEventType.PROCESS_COMPONENTS_REQUEST);

    const {failedComponents} = await processComponents(
        componentsWithChanges,
        imageUrls,
        {
            getComponentName,
            getFileName,
            getSvgrConfig,
            getSvgoConfig,
            exportMode,
            outputDirectory,
            exportPngTemplate,
        },
    );

    emit(EEventType.PROCESS_COMPONENTS_SUCCESS);

    emit(EEventType.PROCESS_META_REQUEST);

    const componentsWithoutErrors = differenceBy(
        components,
        failedComponents,
        component => component.key,
    );
    const meta = collectMeta(componentsWithoutErrors);

    await metaFileProvider.save(meta);
    emit(EEventType.PROCESS_META_SUCCESS);

    return {hasErrors: failedComponents.length > 0};
}
