import path from 'path';
import fse from 'fs-extra';
import {FullComponentMetadata} from 'figma-js';

import {TImagesUrl} from '../../types/TImagesUrl';
import IFriggaOptions from '../../../types/IFriggaOptions';

import fetchComponentSource from './utilities/fetchComponentSource/fetchComponentSource';
import convertComponent from './utilities/convertComponent/convertComponent';

import {EEventType, emit} from '../../../eventEmitter';

interface IProcessComponentsResult {
    failedComponents: FullComponentMetadata[];
}

export default async function processComponents(
    components: FullComponentMetadata[],
    images: TImagesUrl,
    {
        getComponentName,
        getSvgrConfig,
        getSvgoConfig,
        exportMode,
        getFileName,
        outputDirectory,
        exportPngTemplate,
    }: Required<
        Pick<
            IFriggaOptions,
            | 'getComponentName'
            | 'getSvgrConfig'
            | 'getSvgoConfig'
            | 'exportMode'
            | 'getFileName'
            | 'outputDirectory'
        >
    > &
        Pick<IFriggaOptions, 'exportPngTemplate'>,
): Promise<IProcessComponentsResult> {
    const failedComponents: FullComponentMetadata[] = [];

    const promises = components.map(component =>
        (async (): Promise<void> => {
            const {name, key, file_key: fileKey, node_id: nodeId} = component;
            const icon = {name, key};
            const url = images[fileKey][nodeId];

            if (!url) {
                return;
            }

            try {
                emit(EEventType.PROCESS_COMPONENT_REQUEST, icon);

                const fileName = getFileName(icon, exportMode);
                const filePath = path.join(outputDirectory, fileName);

                const source = await fetchComponentSource({
                    url,
                    filePath,
                    exportMode,
                });

                const code = await convertComponent(source, icon, {
                    getComponentName,
                    getSvgoConfig,
                    getSvgrConfig,
                    exportMode,
                    exportPngTemplate,
                });

                await fse.outputFile(filePath, code);

                emit(EEventType.PROCESS_COMPONENT_SUCCESS, icon);
            } catch (e) {
                emit(EEventType.PROCESS_COMPONENT_FAILED, {
                    icon,
                    error: e as Error,
                });

                failedComponents.push(component);
            }
        })(),
    );

    await Promise.all(promises);

    return {failedComponents};
}
