import axios from 'axios';
import fse from 'fs-extra';
import * as stream from 'stream';
import * as path from 'path';
import {promisify} from 'util';

import EExportMode from '../../../../../types/EExportMode';

import setAxiosRetry from '../../../setAxiosRetry';

interface IGetComponentSourceParams {
    url: string;
    filePath: string;
    exportMode: EExportMode;
}

/**
 * Получения исходников изображения из FigmaAPI
 * - для EExportMode.SVG и EExportMode.REACT_COMPONENT возвращает текстовое содержимое изображения в svg формате
 * - для EExportMode.PNG скачивает и сохраняет png файл и возвращает имя png файла
 * @param url
 * @param filePath - путь до результирующего компонента, расширение может быть любое jsx, tsx
 * @param exportMode
 */
export default async function fetchComponentSource({
    url,
    filePath,
    exportMode,
}: IGetComponentSourceParams): Promise<string> {
    const axiosInstance = axios.create();

    setAxiosRetry(axiosInstance);

    switch (exportMode) {
        case EExportMode.SVG:
        case EExportMode.REACT_COMPONENT: {
            const {data} = await axiosInstance.get<string>(url, {
                responseType: 'text',
            });

            return data;
        }

        case EExportMode.PNG: {
            const finished = promisify(stream.finished);

            const parsedFilePath = path.parse(filePath);

            await fse.ensureDir(parsedFilePath.dir);

            const filePathWithPngExtension = path.format({
                root: parsedFilePath.root,
                dir: parsedFilePath.dir,
                name: parsedFilePath.name,
                ext: '.png',
            });

            const writer = fse.createWriteStream(filePathWithPngExtension, {
                autoClose: true,
            });
            const response = await axiosInstance.get<stream.Readable>(url, {
                responseType: 'stream',
            });

            response.data.pipe(writer);

            await finished(writer);

            return path.format({name: parsedFilePath.name, ext: '.png'});
        }
    }
}
