import _chunk from 'lodash/chunk';
import _pickBy from 'lodash/pickBy';
import _groupBy from 'lodash/groupBy';
import Bottleneck from 'bottleneck';
import {ClientInterface, FullComponentMetadata} from 'figma-js';

import {TImagesUrl} from '../types/TImagesUrl';
import EExportMode from '../../types/EExportMode';

import {EEventType, emit} from '../../eventEmitter';
import getFigmaFormatByExportMode from './getFigmaFormatByExportMode';

const RENDER_RATE_LIMIT = 1000;
const CHUNK_SIZE = 64;

interface IRenderComponentsParams {
    figmaClient: ClientInterface;
    components: FullComponentMetadata[];
    exportMode: EExportMode;
    exportScale: number | undefined;
}

export default async function renderComponents({
    figmaClient,
    components,
    exportMode,
    exportScale,
}: IRenderComponentsParams): Promise<TImagesUrl> {
    const groupedIconsByFile = _groupBy(components, icon => icon.file_key);

    const images: TImagesUrl = {};

    const bottleneck = new Bottleneck({
        minTime: RENDER_RATE_LIMIT,
    });

    const promises = Object.entries(groupedIconsByFile).flatMap(
        ([fileKey, fileComponents]) => {
            const chunks = _chunk(fileComponents, CHUNK_SIZE);

            return chunks.map(chunk => {
                images[fileKey] = images[fileKey] || {};

                return bottleneck.schedule(async () => {
                    chunk.forEach(component => {
                        emit(EEventType.RENDER_COMPONENT_REQUEST, component);
                    });

                    const res = await figmaClient.fileImages(fileKey, {
                        format: getFigmaFormatByExportMode(exportMode),
                        ids: chunk.map(component => component.node_id),
                        scale: exportScale,
                    });

                    const imagesFromNodes = res.data.images;

                    images[fileKey] = {
                        ...images[fileKey],
                        ..._pickBy(imagesFromNodes, item => item),
                    };

                    chunk.forEach(component => {
                        emit(EEventType.RENDER_COMPONENT_SUCCESS, component);
                    });
                });
            });
        },
    );

    await Promise.all(promises);

    return images;
}
