import {Config} from '@svgr/core';
import {OptimizeOptions} from 'svgo';

import IComponentNode from './IComponentNode';
import EExportMode from './EExportMode';

export default interface IFriggaOptions {
    /**
     * Figma file key for search components
     * CLI param name --file or -file
     * @link https://a.yandex-team.ru/arcadia/travel/frontend/frigga/README.md#how-to-get-file-key-and-node-id
     */
    file: string;

    /**
     * Output directory for generated components
     * CLI param name --output or -o
     */
    outputDirectory: string;

    /**
     * Figma auth token
     * CLI param name --token or -t
     * Environment variable FIGMA_AUTH_TOKEN
     * Setting via config file is not secured and not supported
     * @link https://a.yandex-team.ru/arcadia/travel/frontend/frigga/README.md#how-to-get-figma-auth-token
     */
    authToken: string;

    /**
     * Figma API host
     * @default 'api.figma.com'
     */
    apiHost: string;

    /**
     * Regenerate file without changes
     * @default false
     */
    noCache: boolean;

    /**
     * Export mode
     * svg - use SVGO
     * react-component - use SVGR
     * png
     * @default react-component
     */
    exportMode?: EExportMode;

    /**
     * Filter function for components
     * Skip component when function return false
     */
    select: (meta: IComponentNode) => boolean;

    /**
     * Customize filename for each component
     */
    getFileName: (meta: IComponentNode, exportMode: EExportMode) => string;

    /**
     * Customize component name for SVGR
     * Work only in 'react-component' export mode
     */
    getComponentName?: (meta: IComponentNode) => string;

    /**
     * Customize params for SVGR
     * Work only in 'react-component' export mode
     * @link https://react-svgr.com/docs/options/
     */
    getSvgrConfig?: (meta: IComponentNode) => Config;

    /**
     * Customize params for SVGO
     * Work only in 'svg' export mode
     * @link https://github.com/svg/svgo/#configuration
     */
    getSvgoConfig?: (meta: IComponentNode) => OptimizeOptions;

    /**
     * Customize scale for export raster images
     * A number between 0.01 and 4, the image scaling factor
     */
    exportScale?: number;

    /**
     * Template for image component
     * Required strings in template for replace
     * «IMAGE_PATH» - png image path
     * «IMAGE_COMPONENT_NAME» - human readable component name
     */
    exportPngTemplate?: string;
}
