import {camelCase, upperFirst} from 'lodash';
import getFunctionName from '@yandex-data-ui/i18n-ts/client/getFunctionName';

import {getLiteralOrIdentifierValue} from '../utilities/getLiteralOrIdentifierValue';

export default function ymapsTanker(file, api) {
    const j = api.jscodeshift;
    const ast = j(file.source);

    if (!ast.find(j.ImportDeclaration, {source: {value: 'i18n'}}).length) {
        return ast.toSource();
    }

    const keysets = {};
    const mapKeySetNameToI18nBlockName = {};

    let hasDynamicKeyset = false;

    ast.find(j.CallExpression, {callee: {name: 'i18n'}}).replaceWith(path => {
        const {node} = path;

        const keySetName = getLiteralOrIdentifierValue(
            j,
            ast,
            node.arguments[0],
        );

        if (!keySetName) {
            throw new Error('Unsupported keySetName');
        }

        const keyName = getLiteralOrIdentifierValue(j, ast, node.arguments[1]);

        const blockName = Object.keys(mapKeySetNameToI18nBlockName).length
            ? `i18n${upperFirst(camelCase(keySetName))}Block`
            : 'i18nBlock';

        mapKeySetNameToI18nBlockName[keySetName] =
            mapKeySetNameToI18nBlockName[keySetName] || blockName;

        const finallyBlockName = mapKeySetNameToI18nBlockName[keySetName];

        if (!keyName) {
            hasDynamicKeyset = true;

            return j.callExpression(
                j.callExpression(
                    j.identifier('unsafeDynamicKeyset'),
                    [j.identifier(finallyBlockName), node.arguments[1]].filter(
                        Boolean,
                    ),
                ),
                [node.arguments[2]].filter(Boolean),
            );
        }

        if (!keysets[keySetName]) {
            keysets[keySetName] = new Set();
        }

        const functionName = getFunctionName(keyName);

        node.arguments = [node.arguments[2]].filter(Boolean);

        node.callee = j.memberExpression(
            j.identifier(finallyBlockName),
            j.identifier(functionName),
        );

        return node;
    });

    ast.find(j.ImportDeclaration, {source: {value: 'i18n'}}).replaceWith(() => {
        return [
            ...(hasDynamicKeyset
                ? [
                      j.importDeclaration(
                          [
                              j.importSpecifier(
                                  j.identifier('unsafeDynamicKeyset'),
                              ),
                          ],
                          j.literal('utilities/tanker/unsafeDynamicKeyset'),
                      ),
                  ]
                : []),
            ...Object.entries(mapKeySetNameToI18nBlockName).map(
                ([keysetName, variableName]) => {
                    return j.importDeclaration(
                        [
                            j.importNamespaceSpecifier(
                                j.identifier(variableName),
                            ),
                        ],
                        j.literal(`i18n/${keysetName}`),
                    );
                },
            ),
        ];
    });

    return ast.toSource({quote: 'single'});
}

export const parser = 'tsx';
