import fs from 'fs-extra';
import path from 'path';

import IBuilderOptions from '../types/IBuilderOptions';
import ITranslations from '../types/ITranslations';
import ELanguages from '../types/ELanguages';
import isDefined from '../types/isDefined';
import ITankerLinkOptions from '../types/ITankerLinkOptions';

import getFunctionName from '../utilities/getFunctionName/getFunctionName';

import KeyPreparer from './KeyPreparer/KeyPreparer';
import KeyFunctionBuilder from './KeyFunctionBuilder/KeyFunctionBuilder';
import JSDocBuilder from './JSDocBuilder/JSDocBuilder';
import TankerLinkBuilder from './TankerLinkBuilder/TankerLinkBuilder';
import OldFunctionBuilder from './OldFunctionBuilder/OldFunctionBuilder';

export default class Builder {
    private readonly outFolder: string;
    private readonly originalLang: string;
    private readonly tankerLinkOptions: ITankerLinkOptions | undefined;
    private readonly langConstantName: string;
    private readonly prepareKey: (key: string) => string;
    private readonly i18nBackwardCompatibility: boolean;
    private readonly i18nBackwardCompatibilityFileName: string;

    constructor({
        outFolder,
        originalLang = ELanguages.RU,
        tankerLinkOptions,
        langConstantName = '__BUILD_LANG__',
        prepareKey = getFunctionName,
        i18nBackwardCompatibility = false,
        i18nBackwardCompatibilityFileName = 'index',
    }: IBuilderOptions) {
        this.outFolder = outFolder;
        this.originalLang = originalLang;
        this.tankerLinkOptions = tankerLinkOptions;
        this.langConstantName = langConstantName;
        this.prepareKey = prepareKey;
        this.i18nBackwardCompatibility = i18nBackwardCompatibility;
        this.i18nBackwardCompatibilityFileName =
            i18nBackwardCompatibilityFileName;
    }

    async build(translations: ITranslations): Promise<void> {
        await fs.ensureDir(this.outFolder);

        const keySets = Object.values(translations.keySets);

        const keySetNames = keySets.map(keySet => keySet.name);

        await this.clearKeySetFiles(keySetNames);

        const tasks = keySets.map(keySet => {
            return (async (): Promise<void> => {
                const keys = Object.values(keySet.keys);

                const tsKeys = keys.map(key => {
                    const preparedTranslationEntries = Object.values(
                        key.translations,
                    )
                        .filter(isDefined)
                        .map(translation => {
                            try {
                                const keyPreparer = new KeyPreparer();

                                return keyPreparer.prepare(translation);
                            } catch (e) {
                                throw new Error(
                                    `Error while processing keySet: ${keySet.name}, key: ${key.name}`,
                                );
                            }
                        });

                    const originalTranslation = preparedTranslationEntries.find(
                        t => t.lang === this.originalLang,
                    );

                    if (!originalTranslation) {
                        throw new Error(
                            'Not translation for original language',
                        );
                    }

                    const preparedTranslations =
                        preparedTranslationEntries.reduce(
                            (acc, preparedTranslation) => {
                                return {
                                    ...acc,
                                    [preparedTranslation.lang]:
                                        preparedTranslation.body,
                                };
                            },
                            {},
                        );

                    const keyFunctionBuilder = new KeyFunctionBuilder({
                        langConstantName: this.langConstantName,
                    });

                    return {
                        tsCode: keyFunctionBuilder.build(
                            this.prepareKey(key.name),
                            originalTranslation.params,
                            preparedTranslations,
                        ),
                        descriptionText:
                            originalTranslation.sourceText ||
                            originalTranslation.text,
                        keySetName: keySet.name,
                        keyName: key.name,
                    };
                });

                const jsDocBuilder = new JSDocBuilder();
                const tankerLinkBuilder = this.tankerLinkOptions
                    ? new TankerLinkBuilder(this.tankerLinkOptions)
                    : null;
                const keySetLink = tankerLinkBuilder
                    ? {
                          url: tankerLinkBuilder.build(keySet.name),
                          name: 'Tanker',
                      }
                    : undefined;

                const result =
                    `/* global ${this.langConstantName} */\n` +
                    `${jsDocBuilder.build({
                        link: keySetLink,
                    })}\n\n` +
                    tsKeys.reduce(
                        (
                            acc,
                            {tsCode, descriptionText, keySetName, keyName},
                        ) => {
                            const keyLink = tankerLinkBuilder
                                ? {
                                      url: tankerLinkBuilder.build(
                                          keySetName,
                                          keyName,
                                      ),
                                      name: 'Tanker',
                                  }
                                : undefined;

                            return (
                                acc +
                                jsDocBuilder.build({
                                    description: descriptionText,
                                    link: keyLink,
                                }) +
                                '\n' +
                                tsCode +
                                '\n\n'
                            );
                        },
                        '',
                    );

                const filePath = path.join(this.outFolder, `${keySet.name}.ts`);

                await fs.writeFile(filePath, result, 'utf8');
            })();
        });

        if (this.i18nBackwardCompatibility) {
            tasks.push(
                (async (): Promise<void> => {
                    const oldFunctionBuilder = new OldFunctionBuilder();

                    const filePath = path.join(
                        this.outFolder,
                        `${this.i18nBackwardCompatibilityFileName}.ts`,
                    );

                    await fs.writeFile(
                        filePath,
                        oldFunctionBuilder.build(translations),
                        'utf8',
                    );
                })(),
            );
        }

        await Promise.all(tasks);
    }

    /**
     * Удаляет ts файлы для несуществующих кейсетов
     * @param keySetNames список актуальных кейсетов
     */
    private async clearKeySetFiles(keySetNames: string[]): Promise<void> {
        const set = new Set(keySetNames);

        const keySetFiles = await fs.readdir(this.outFolder);

        const filesForRemoval = keySetFiles.filter(
            fileName => !set.has(fileName),
        );

        await Promise.all(
            filesForRemoval.map(fileName => {
                const filePath = path.join(this.outFolder, fileName);

                return fs.unlink(filePath);
            }),
        );
    }
}
