import {AstBuilder, Compiler, Parser} from 'compact-tl';
import type {Processor} from 'compact-tl';
import {decode} from 'html-entities';

import ELanguages from '../../types/ELanguages';

interface IOptions {
    lang: ELanguages;
}

/**
 * Генерирует js функции из шаблонов танкера
 */
export default class ContentBuilder {
    private ast: AstBuilder;
    private compiler: Compiler;
    private options: IOptions;

    constructor(options: IOptions) {
        this.options = options;

        this.ast = new AstBuilder();
        this.compiler = new Compiler();

        this.ast.registerBlockSection('if', 'else');
        this.compiler.registerBlockHelper('if', this.blockHelperIf);

        this.compiler.registerHelper('plural', this.helperPlural);
    }

    compile(input: string): string {
        const ast = this.ast.build(new Parser(decode(input)));

        return this.compiler.compile(ast);
    }

    private blockHelperIf: Processor = (ast, compiler) => {
        const elseSection =
            ast.sections[0] && ast.sections[0].name === 'else'
                ? ast.sections[0]
                : null;

        return (
            '(' +
            compiler.generateExpression(ast.arguments.list[0]) +
            '?' +
            compiler.compileThread(ast.mainSection) +
            ':' +
            (elseSection ? compiler.compileThread(elseSection) : '""') +
            ')'
        );
    };

    private helperPlural: Processor = ast => {
        const args = ast.arguments.hash;
        const lastDigit = `params.${args.count.name} % 10`;
        const tens = `params.${args.count.name} % 100`;

        let result =
            this.options.lang === ELanguages.EN
                ? `params.${args.count.name} === 1 ? ${JSON.stringify(
                      args.one.value,
                  )} :`
                : `${lastDigit} === 1 && ${tens} !== 11 ? ${JSON.stringify(
                      args.one.value,
                  )} :`;

        const someMany = [args.some, args.many].reduce((acc, item) => {
            const value = item && item.value && JSON.stringify(item.value);

            return value && acc.indexOf(value) === -1 ? acc.concat(value) : acc;
        }, []);

        if (someMany.length === 1) {
            result += someMany[0];
        } else {
            result += `(${lastDigit} > 1 && ${lastDigit} < 5 && (${tens} < 10 || ${tens} > 20) ? ${someMany[0]} : ${someMany[1]})`;
        }

        return `(${result})`;
    };
}
