import * as ts from 'typescript';
import {isEmpty} from 'lodash';

import {EParamType, TParams} from '../types/TParams';
import ELanguages from '../../types/ELanguages';

interface IKeyFunctionBuilderOptions {
    /**
     * Имя переменной по которой выбирается язык при сборке
     */
    langConstantName: string;
}

/**
 * Собирает ts функцию для переводов нескольких языков
 */
export default class KeyFunctionBuilder {
    private static buildParamsPropertyType(paramType: EParamType): ts.TypeNode {
        if (paramType === EParamType.NUMBER) {
            return ts.factory.createKeywordTypeNode(
                ts.SyntaxKind.NumberKeyword,
            );
        }

        if (paramType === EParamType.STRING) {
            return ts.factory.createKeywordTypeNode(
                ts.SyntaxKind.StringKeyword,
            );
        }

        return ts.factory.createUnionTypeNode([
            ts.factory.createKeywordTypeNode(ts.SyntaxKind.NumberKeyword),
            ts.factory.createKeywordTypeNode(ts.SyntaxKind.StringKeyword),
            ts.factory.createKeywordTypeNode(ts.SyntaxKind.BooleanKeyword),
        ]);
    }

    private static getTemplatePlaceholder(lang: ELanguages): string {
        return `__${lang}_CODE_PLACEHOLDER__`;
    }

    private readonly langConstantName: string;

    constructor({langConstantName}: IKeyFunctionBuilderOptions) {
        this.langConstantName = langConstantName;
    }

    build(
        keyName: string,
        params: TParams,
        translations: Partial<Record<ELanguages, string>>,
    ): string {
        const resultFile = ts.createSourceFile(
            '',
            '',
            ts.ScriptTarget.Latest,
            false,
            ts.ScriptKind.TS,
        );
        const printer = ts.createPrinter({newLine: ts.NewLineKind.LineFeed});

        const tsCode = printer.printNode(
            ts.EmitHint.Unspecified,
            this.buildAST(keyName, params, translations),
            resultFile,
        );

        return this.replacePlaceholders(tsCode, translations);
    }

    private buildParamsType(params: TParams): ts.ParameterDeclaration[] {
        if (isEmpty(params)) {
            return [];
        }

        return [
            ts.factory.createParameterDeclaration(
                undefined,
                undefined,
                undefined,
                ts.factory.createIdentifier('params'),
                undefined,
                ts.factory.createTypeLiteralNode(
                    Object.entries(params).map(([paramName, paramType]) =>
                        ts.factory.createPropertySignature(
                            undefined,
                            ts.factory.createIdentifier(paramName),
                            undefined,
                            KeyFunctionBuilder.buildParamsPropertyType(
                                paramType,
                            ),
                        ),
                    ),
                ),
                undefined,
            ),
        ];
    }

    private buildAST(
        keyName: string,
        params: TParams,
        translations: Partial<Record<ELanguages, string>>,
    ): ts.FunctionDeclaration {
        return ts.factory.createFunctionDeclaration(
            undefined,
            [ts.factory.createModifier(ts.SyntaxKind.ExportKeyword)],
            undefined,
            ts.factory.createIdentifier(keyName),
            undefined,
            this.buildParamsType(params),
            ts.factory.createKeywordTypeNode(ts.SyntaxKind.StringKeyword),
            ts.factory.createBlock(
                [
                    ts.factory.createSwitchStatement(
                        ts.factory.createIdentifier(this.langConstantName),
                        ts.factory.createCaseBlock([
                            ...Object.entries(translations)
                                .filter(([, code]) => code)
                                .map(([lang]) =>
                                    ts.factory.createCaseClause(
                                        ts.factory.createStringLiteral(lang),
                                        [
                                            ts.factory.createBlock(
                                                [
                                                    ts.factory.createReturnStatement(
                                                        ts.factory.createIdentifier(
                                                            KeyFunctionBuilder.getTemplatePlaceholder(
                                                                lang as ELanguages,
                                                            ),
                                                        ),
                                                    ),
                                                ],
                                                true,
                                            ),
                                        ],
                                    ),
                                ),
                            ts.factory.createDefaultClause([
                                ts.factory.createBlock(
                                    [
                                        ts.factory.createThrowStatement(
                                            ts.factory.createNewExpression(
                                                ts.factory.createIdentifier(
                                                    'Error',
                                                ),
                                                undefined,
                                                [
                                                    ts.factory.createStringLiteral(
                                                        `Unexpected ${this.langConstantName}`,
                                                    ),
                                                ],
                                            ),
                                        ),
                                    ],
                                    true,
                                ),
                            ]),
                        ]),
                    ),
                ],
                true,
            ),
        );
    }

    private replacePlaceholders(
        tsCode: string,
        translations: Partial<Record<ELanguages, string>>,
    ): string {
        return Object.entries(translations).reduce((result, [lang, code]) => {
            if (!code) {
                return result;
            }

            return result.replace(
                KeyFunctionBuilder.getTemplatePlaceholder(lang as ELanguages),
                code,
            );
        }, tsCode);
    }
}
